﻿///
/// Generic function to access and save settings
///
var mouseOverHideIf = false;
function getSettings(_container, _name, _ix) {
    var _value = $(_container).find(':input[data-name=' + _name + ']').val();
    var _data = JSON.parse(_value);

    if (_data === null) {
        _data = [];
    }

    if (_ix === undefined || isNaN(_ix)) {
        return _data;
    } else {
        return _data[_ix];
    }
}

function setSettings(_container, _name, _value, _ix) {
    var _dataField = $(_container).find(':input[data-name=' + _name + ']');
    var _data = JSON.parse(_dataField.val());

    if (_ix === undefined || isNaN(_ix)) {
        _dataField.val(JSON.stringify(_value));
    } else {
        _data[_ix] = _value;
        _dataField.val(JSON.stringify(_data));
    }

    return true;
}

function getQuestionEditPanel(_element) {
    return $(_element).closest('.v-questionEdit');
}

function getOptionPanel(_element) {
    var _questionPanel = getQuestionEditPanel(_element);
    return _questionPanel ? $('.question-main-options', _questionPanel) : null;
}


function initGridChoiceColumn(frame, choiceIndex) {
    var editor = new GridEditor(questionnaireEditor.currentEditor);
    var column = _.first(editor.getColumnSettings());
    InitializeGridColumnChoiceEditorValues(frame, column, column.Choices[choiceIndex], editor);
}

function saveGridChoiceColumn(frame, choiceIndex) {
    var editor = new GridEditor(questionnaireEditor.currentEditor);
    var column = _.first(editor.getColumnSettings())
    RetrieveGridColumnChoiceEditorValues(frame, column, column.Choices[choiceIndex]);
    editor.setColumnSettings([column]);
    editor.Container.gridEditor("refreshTable");
    return modalDialog.close();
}

function initGridColumn(frame, columnIndex) {
    var content = $(frame).contents();
    var editor = new GridEditor(questionnaireEditor.currentEditor);
    var cell = questionnaireEditor.currentEditor.find('thead td[data-column=' + columnIndex + ']');
    var column = editor.getColumnSettings(cell);

    if (questionnaireEditor.currentEditor) {
        var gridType = questionnaireEditor.currentEditor.attr('data-grid-type') || GridEditor.hybridGridType;

        if (gridType === GridEditor.runningTotalGridType) {
            // For the running total, we add the following attributes
            var rtAttributes = ['LowerBoundNotFulfilledMessage', 'UpperBoundNotFulfilledMessage'];

            $('.v-columnTypeSetting .v-setting, .v-columnTypeSetting .v-translatableSetting', content).each(function () {
                if (!$(this).closest('.v-columnTypeSetting').hasClass('type' + column.ColumnType)) {
                    if (!_.include(rtAttributes, $(this).attr('data-name'))) {
                        $(this).closest('.v-columnSetting,.v-columnTypeSetting').remove();
                    } else {
                        $(this).closest('.v-columnTypeSetting').addClass('type' + column.ColumnType);
                    }
                }
            });

            $('.v-setting[data-name=ColumnType]', content).closest('.v-columnSetting').remove();
        }
    }
    InitializeGridColumnEditorValues(frame, column, editor);
}

function saveGridColumn(frame, columnIndex) {
    var editor = new GridEditor(questionnaireEditor.currentEditor);
    var cell = questionnaireEditor.currentEditor.find('thead td[data-column=' + columnIndex + ']');
    questionnaireEditor.errorOnRequest = false;

    var column = RetrieveGridColumnEditorValues(frame, editor.getColumnSettings(cell));

    if (!questionnaireEditor.validateSettings(editor.Container, column.ColumnType, column.TypeSettings, column.Choices)) {
        return false;
    }

    editor.setColumnSettings(column, cell);
    editor.Container.gridEditor("refreshTable");
    return modalDialog.close();
}

function GridEditor(_container) {
    this.Container = _container;

    // Adds the original column positions
    var _columnPositions = _.map(_.compact(this.getColumnSettings()), function (_value) { return _value.Position; });
    var _columnPositionSet = _.size(_columnPositions) === _.size(_.unique(_columnPositions));
    if (!_columnPositionSet) {
        this.setColumnSettings($.map(this.getColumnSettings(), function (_value, _ix) {
            _value.Position = _ix;
            return _value;
        }));
    }

    // Adds the original row positions
    var _rowPositions = _.map(_.compact(this.getRowSettings()), function (_value) { return _value.Position; });
    var _rowPositionSet = _.size(_rowPositions) === _.size(_.unique(_rowPositions));
    if (!_rowPositionSet) {
        this.setRowSettings($.map(this.getRowSettings(), function (_value, _ix) {
            _value.Position = _ix;
            return _value;
        }));
    }

    return this;
}

// The container is the HTML object that contains editor. It should be used to find HTML objects within the page
GridEditor.prototype.Container = null;

// This flag is for new grid. We allow the variable name to be changed when the question name changes
// If we can rename variable, this will be disabled as soon as a variable is manually renamed
GridEditor.prototype.CanRenameVariables = false;

GridEditor.prototype.getRowSettings = function (_row) { return getSettings(this.Container, 'rows', this.getRowIndex(_row)); };
GridEditor.prototype.setRowSettings = function (_value, _row) { return setSettings(this.Container, 'rows', _value, this.getRowIndex(_row)); };
GridEditor.prototype.getColumnSettings = function (_col) { return getSettings(this.Container, 'columns', this.getColumnIndex(_col)); };
GridEditor.prototype.setColumnSettings = function (_value, _col) { return setSettings(this.Container, 'columns', _value, this.getColumnIndex(_col)); };

GridEditor.prototype.getOrderedColumns = function () { return _.sortBy(_.compact(this.getColumnSettings()), function (column) { return column.Position; }); };
GridEditor.prototype.getAllVariables = function () { return _.compact(_.flatten(_.map(this.getRowSettings(), function (row) { return row && row.RowType == GridEditor.variableRowType && row.VariableNames; }))) };

///
/// Constants
///
GridEditor.titleRowType = "1";
GridEditor.variableRowType = "2";
GridEditor.textRowType = "3";
GridEditor.choiceRowType = "4";


GridEditor.hybridGridType = "HybridGrid";
GridEditor.choiceGridType = "ChoiceGrid";
GridEditor.imageChoiceGridType = "ImageChoiceGrid";
GridEditor.runningTotalGridType = "RunningTotal";
GridEditor.starRatingGridType = "StarRatingGrid";
GridEditor.netPromoterGridType = "NetPromoter";


///
/// Column functions
///

GridEditor.prototype.getColumnIndex = function (_col) {
    return Number($(_col).attr('data-column'));
};

GridEditor.prototype.getColumnPosition = function (_col) {
    return Number(this.getColumnSettings(_col).Position);
};

GridEditor.prototype.numCells = function (column) {
    var editor = this;
    if (column === undefined) {
        return _.reduce(_.compact(editor.getColumnSettings()),
            function (accumulator, col) { return accumulator + editor.numCells(col); },
            0);
    } else {
        return (_.include(['RadioButton', 'CheckBox', 'ImageSelector'], column.ColumnType) && _.size(column.Choices)) || 1;
    }
};

GridEditor.prototype.numLabels = function (row, column) {
    if (row.RowType == GridEditor.textRowType) {
        return this.numCells(column);
    } else if (row.RowType == GridEditor.titleRowType) {
        return 1;
    } else {
        return 0;
    }
};

GridEditor.prototype.fixRowLabels = function (row) {
    var expectedNumLabels;
    var editor = this;

    if (row.RowType == GridEditor.textRowType) {
        row.Labels = row.Labels || [];

        if (row.MergeCells) {
            expectedNumLabels = [1];
        } else {
            expectedNumLabels = _.map(this.getOrderedColumns(), function (column) { return editor.numCells(column); });
        }

        while (_.size(row.Labels) < _.size(expectedNumLabels)) {
            row.Labels.push([]);
        }

        for (var i = 0; i < _.size(expectedNumLabels); i++) {
            row.Labels[i] = row.Labels[i] || [];

            if (_.isString(row.Labels[i])) { row.Labels[i] = [row.Labels[i]]; } // Should be an array

            var numColumnLabels = expectedNumLabels[i];
            while (_.size(row.Labels[i]) < numColumnLabels) {
                row.Labels[i].push('');
            }
        }

    } else if (row.RowType == GridEditor.titleRowType) {
        row.Labels = _.map(row.Labels || [], function (label) { return _.isArray(label) ? label[0] : label; });

        if (row.MergeCells) {
            expectedNumLabels = 1;
        } else {
            expectedNumLabels = _.size(this.getOrderedColumns());
        }

        while (_.size(row.Labels) < expectedNumLabels) {
            row.Labels.push('');
        }
    }

};

GridEditor.prototype.getColumnLabelPosition = function (row, col) {

    if (row && (row.RowType == GridEditor.titleRowType || row.RowType == GridEditor.textRowType) && !row.MergeCells) {
        return col.Position;
    }

    return null;
};

GridEditor.prototype.moveChoice = function (columnIndex, choiceIndex, displacement) {

    choiceIndex = Number(choiceIndex);

    var columns = this.getColumnSettings();
    var column = columns[columnIndex];

    var choice = column.Choices[choiceIndex];
    column.Choices.splice(choiceIndex, 1);
    column.Choices.splice(choiceIndex + displacement, 0, choice);
    this.setColumnSettings(columns);

    var rows = this.getRowSettings();
    _.each(rows, function (row) {
        if (row && row.Labels && _.isArray(row.Labels[columnIndex]) && !row.MergeCells) {
            var label = row.Labels[columnIndex][choiceIndex];
            row.Labels[columnIndex].splice(choiceIndex, 1);
            row.Labels[columnIndex].splice(choiceIndex + displacement, 0, label);
        }
    });

    this.setRowSettings(rows);

};

GridEditor.prototype.moveColumn = function (_col, targetPosition) {

    var editor = this;
    var currentColumn = this.getColumnSettings(_col);


    var _columns = this.getColumnSettings();
    var _index = this.getColumnIndex(_col);

    var _minPosition = 0;
    var _maxPosition = _.max(_.pluck(_.compact(_columns), 'Position'));

    var fromPosition = this.getColumnPosition(_col);
    if (fromPosition < targetPosition) {
        targetPosition--;
    }
    var displacement = targetPosition - fromPosition;


    _columns = _.map(_columns, function (_value, _ix) {
        if (_value) {
            if (_ix == _index) {
                _value.Position = targetPosition;
            } else {
                if (_value.Position > fromPosition) {
                    _value.Position--;
                }
                if (_value.Position >= targetPosition) {
                    _value.Position++;
                }
            }

            _value.Position = Math.max(_minPosition, Math.min(_maxPosition, _value.Position));
        }
        return _value;
    });


    // Modify the row labels -- processing over the unmoved columns
    var rows = this.getRowSettings();
    for (var rowIndex = 0; rowIndex < rows.length; rowIndex++) {
        if (rows[rowIndex] && rows[rowIndex].Labels) {
            // Find the label columns that are part of the source position
            var startLabelIndex = editor.getColumnLabelPosition(rows[rowIndex], currentColumn);
            if (startLabelIndex !== null) {
                this.fixRowLabels(rows[rowIndex]);
                var labelsToMove = rows[rowIndex].Labels.splice(startLabelIndex, 1);
                var newLabelIndex = startLabelIndex + displacement;
                Array.prototype.splice.apply(rows[rowIndex].Labels, [newLabelIndex, 0].concat(labelsToMove));
            }
        }
    }

    this.setColumnSettings(_columns);
    this.setRowSettings(rows);

};

GridEditor.prototype.generateVariableName = function (columnCode, rowIndex, rowIndexForDisplay) {

    if (rowIndexForDisplay === undefined) {
        rowIndexForDisplay = rowIndex;
    }

    var allVariableNames = _.compact(_.flatten(
        _.pluck(_.compact(_.reject(this.getRowSettings(), function (row, index) { return index == rowIndex; })),
            "VariableNames")));

    var variableName = (GetQuestionName(this.Container) + "_" + columnCode + (rowIndexForDisplay + 1).toString()).toUpperCase();
    if (_.include(allVariableNames, variableName)) {
        return this.generateVariableName(columnCode, rowIndex, rowIndexForDisplay + 1);
    } else {
        return variableName;
    }
};

GridEditor.prototype.isColumnCodeAvailable = function (columnCode, columnIndex) {
    var rows = this.getRowSettings();
    var allVariables = _.compact(_.flatten(_.map(rows, function (row) { return row && row.RowType == GridEditor.variableRowType && row.VariableNames; })));

    if (columnIndex !== undefined) {
        columnVariables = _.map(rows, function (row) { return row && row.RowType == GridEditor.variableRowType && row.VariableNames && row.VariableNames[columnIndex]; });
        allVariables = _.filter(allVariables, function (v) { return !_.include(columnVariables, v); });
    }

    columnNamePrefix = GetQuestionName(this.Container).toUpperCase() + "_" + columnCode;
    columnCodeUsed = new RegExp('^' + columnNamePrefix);
    return !_.any(allVariables, function (name) { return columnCodeUsed.test(name); });
};

GridEditor.prototype.generateColumnCode = function (column, index) {
    var position = column.Position || 0;

    function getAlphaCode(p) {
        if (p === 0) {
            return "";
        } else if (p > 0 && p < 27) {
            return String.fromCharCode(64 + p);
        } else {
            return getAlphaCode(Math.floor(p / 25)) + getAlphaCode(p % 25);
        }
    }


    do {
        position++;
        columnCode = getAlphaCode(position);
    } while (!this.isColumnCodeAvailable(columnCode, index));

    return columnCode;
};

GridEditor.prototype.addColumn = function (label, columnType) {
    var columns = this.getColumnSettings();
    var position = _.max(_.pluck(_.compact(columns), 'Position').concat([-1])) + 1;

    // Adding a column setting
    var column = { Label: cleanHtml(label), ColumnType: columnType || 'Text', Position: position, AnswerRequired: true };

    // Adding variables for the column in each row
    var editor = this;
    var columnCode = editor.generateColumnCode(column);
    var rows = editor.getRowSettings();
    var rowIndex = 0;
    rows = _.map(rows, function (row, ix) {
        if (row && row.RowType == GridEditor.variableRowType) {
            row.VariableNames = row.VariableNames || [];
            row.VariableNames.push(editor.generateVariableName(columnCode, rowIndex));

            row.VariableHideIfs = row.VariableHideIfs || [];
            row.VariableHideIfs.push(null);
            rowIndex++;
        }
        else if (row && row.RowType == GridEditor.titleRowType && !row.MergeCells) {
            editor.fixRowLabels(row);
            if (row.Labels.length < column.Position) {
                row.Labels.push(column.Label);
            } else {
                row.Labels[column.Position] = column.Label;
            }
        }
        return row;
    });

    columns.push(column);
    this.setColumnSettings(columns) && this.setRowSettings(rows);
    return column;
};


GridEditor.prototype.deleteColumn = function (_columnHeader) {

    var _columns = this.getColumnSettings();
    var _columnIndex = this.getColumnIndex(_columnHeader);
    var column = _columns[_columnIndex];

    // Remove the column
    _columns[_columnIndex] = null;



    // Adding variables for the column in each row
    var _editor = this;
    var _rows = _.map(this.getRowSettings(), function (_row) {
        if (_row && _row.VariableNames) {
            _row.VariableNames[_columnIndex] = null;
            _row.VariableIds[_columnIndex] = null;
        }

        if (_row && _row.VariableHideIfs) {
            _row.VariableHideIfs[_columnIndex] = null;
        }
        return _row;
    });

    // Removes label associated with that column
    _.each(_rows, function (row) {
        var labelIndex = _editor.getColumnLabelPosition(row, column);
        if (labelIndex >= 0 && row.Labels && row.Labels[labelIndex]) {
            row.Labels.splice(labelIndex, 1);
        }
    });


    // Removing ending null columns
    while (_columns.length > 0 && !_.last(_columns)) {
        _.each(_rows, function (_row) {
            if (_row && _row.VariableNames && _row.VariableNames.length == _columns.length) {
                _row.VariableNames.pop();
            }

            if (_row && _row.VariableHideIfs && _row.VariableHideIfs.length == _columns.length) {
                _row.VariableHideIfs.pop();
            }
        });
        _columns.pop();
    }

    // Adjust position on grid
    var _columnPosition = this.getColumnPosition(_columnHeader);
    _.each(_columns, function (_column) {
        if (_column && _column.Position > _columnPosition) {
            _column.Position--;
        }
    });


    this.setRowSettings(_rows);
    return this.setColumnSettings(_columns);
};

///
/// Row functions
///
GridEditor.prototype.getRowIndex = function (_row) {
    return Number($(_row).attr('data-row'));
};

GridEditor.prototype.getRowPosition = function (_row) {
    return Number(this.getRowSettings(_row).Position);
};

GridEditor.prototype.moveRow = function (_row, _targetPosition, _dragging) {

    var _rows = this.getRowSettings();
    var _rowIndex = this.getRowIndex(_row);

    var _minPosition = 0;
    var _maxPosition = _.max(_.pluck(_.compact(_rows), 'Position'));

    var _fromPosition = this.getRowPosition(_row);
    if (_dragging && _fromPosition < _targetPosition) {
        _targetPosition--;
    }

    _rows = _.map(_rows, function (_value, _ix) {
        if (_value) {
            if (_ix == _rowIndex) {
                _value.Position = _targetPosition;
            } else {
                if (_value.Position > _fromPosition) {
                    _value.Position--;
                }
                if (_value.Position >= _targetPosition) {
                    _value.Position++;
                }
            }

            _value.Position = Math.max(_minPosition, Math.min(_maxPosition, _value.Position));
        }
        return _value;
    });

    this.setRowSettings(_rows);

};

GridEditor.prototype.addContentRow = function () {
    var rows = this.getRowSettings();
    var position = _.max(_.pluck(_.compact(rows), 'Position').concat([-1])) + 1;
    var row = { RowType: GridEditor.titleRowType, Position: position, RandomizationDisabled: true };

    rows.push(row);
    this.setRowSettings(rows);
    return row;
};

GridEditor.prototype.addChoiceRow = function () {
    var rows = this.getRowSettings();
    var position = _.max(_.pluck(_.compact(rows), 'Position').concat([-1])) + 1;
    var row = { RowType: GridEditor.textRowType, Position: position, RandomizationDisabled: true, UseChoiceTexts: true };

    rows.push(row);
    this.setRowSettings(rows);
    return row;
};

GridEditor.prototype.addVariableRow = function (label, hideIfs) {
    var rows = this.getRowSettings();
    var position = _.max(_.pluck(_.compact(rows), 'Position').concat([-1])) + 1;

    var variableNameIndex = _.size(_.filter(rows, function (row) { return row && row.RowType == GridEditor.variableRowType; }));

    var columns = this.getColumnSettings();

    // Adding a row setting
    var row = { RowType: GridEditor.variableRowType, LeftLabel: cleanHtml(label), Position: position, AnswerRequired: true };

    // Adding variables each column of the row
    var editor = this;

    row.VariableNames = _.map(_.reject(columns, function (c) { return !c; }), function (column, columnIndex) {
        var columnCode = editor.generateColumnCode(column, columnIndex);
        return column ? editor.generateVariableName(columnCode, variableNameIndex) : null;
    });

    if (hideIfs && hideIfs.length > 0) {
        row.VariableHideIfs = hideIfs;
    }

    rows.push(row);
    this.setRowSettings(rows);
    return row;
};


GridEditor.prototype.deleteRow = function (_row) {
    $("[data-row=" + _row.attr('data-row') + "]", _row.parent()).remove();
    //$(_row).remove();

    // Put the current row null
    var _rows = this.getRowSettings();
    _rows[this.getRowIndex(_row)] = null;

    // Removing ending rows that are null
    while (_rows.length > 0 && !_.last(_rows)) {
        _rows.pop();
    }

    // Adjust position on grid
    var _rowPosition = this.getRowPosition(_row);
    _.each(_rows, function (_value) {
        if (_value && _value.Position > _rowPosition) {
            _value.Position--;
        }
    });

    return this.setRowSettings(_rows);
};


GridEditor.prototype.resetVariableNames = function () {

    if (this.CanRenameVariables) {
        // Adding variables each column of the row
        var editor = this;

        var rows = this.getRowSettings();
        var columns = this.getColumnSettings();

        // Reset variable names
        _.each(rows, function (row) {
            if (row) {
                if (row.RowType == GridEditor.variableRowType) {
                    row.VariableNames = [];
                } else {
                    delete row.VariableNames;
                }
            }
        });

        _.each(columns, function (column) {
            var columnCode = column ? editor.generateColumnCode(column) : null;
            var rowIndex = 0;
            _.each(_.filter(rows, function (row) { return row && row.RowType == GridEditor.variableRowType; }), function (row) {
                row.VariableNames.push(column ? editor.generateVariableName(columnCode, rowIndex) : null);
                rowIndex++;
            });
        });

        return this.setRowSettings(rows) && this.setColumnSettings(columns);
    } else {
        return false;
    }
};

GridEditor.prototype.createEmptyGrid = function (gridType) {

    this.setRowSettings([]);
    this.setColumnSettings([]);

    switch (gridType) {
        case GridEditor.hybridGridType:
            this.addContentRow();
            this.addChoiceRow();
            var _textColumn = this.addColumn(questionnaireEditor.resources.GridColumnPrefix + '1'); // Text type

            var _checkboxColumn = this.addColumn(questionnaireEditor.resources.GridColumnPrefix + '2', 'CheckBox'); // Checkbox type
            _checkboxColumn.Choices = [{ Code: 1, Text: questionnaireEditor.resources.GridChoicePrefix + '1', Visible: true },
            { Code: 2, Text: questionnaireEditor.resources.GridChoicePrefix + '2', Visible: true },
            { Code: 3, Text: questionnaireEditor.resources.GridChoicePrefix + '3', Visible: true }];

            var _dropdownColumn = this.addColumn(questionnaireEditor.resources.GridColumnPrefix + '3', 'DropDownList'); // Dropdown type
            _dropdownColumn.Choices = [{ Code: 1, Text: $questionnaireEditor.resources.GridChoicePrefix + '1', Visible: true },
            { Code: 2, Text: questionnaireEditor.resources.GridChoicePrefix + '2', Visible: true },
            { Code: 3, Text: questionnaireEditor.resources.GridChoicePrefix + '3', Visible: true }];
            $('#ColumnPrefix', this.Container).text()
            this.setColumnSettings([_textColumn, _checkboxColumn, _dropdownColumn]);
            break;
        case GridEditor.choiceGridType:
        case GridEditor.netPromoterGridType:
            this.addChoiceRow();
            var _column = this.addColumn('', 'RadioButton');
            _column.Choices = [{ Code: 1, Text: questionnaireEditor.resources.GridColumnPrefix + '1', Visible: true },
            { Code: 2, Text: questionnaireEditor.resources.GridColumnPrefix + '2', Visible: true },
            { Code: 3, Text: questionnaireEditor.resources.GridColumnPrefix + '3', Visible: true }];
            this.setColumnSettings([_column]);
            break;
        case GridEditor.imageChoiceGridType:
            this.addChoiceRow();
            var _imageColumn = this.addColumn('', 'ImageSelector');
            _imageColumn.Choices = [{ Code: 1, Text: questionnaireEditor.resources.GridColumnPrefix + '1', Visible: true },
            { Code: 2, Text: questionnaireEditor.resources.GridColumnPrefix + '2', Visible: true },
            { Code: 3, Text: questionnaireEditor.resources.GridColumnPrefix + '3', Visible: true }];
            this.setColumnSettings([_imageColumn]);
            break;
        case GridEditor.runningTotalGridType:
            this.addContentRow();
            this.addColumn(questionnaireEditor.resources.GridColumnPrefix + '1', 'NumericAnswer');
            this.addColumn(questionnaireEditor.resources.GridColumnPrefix + '2', 'NumericAnswer');
            this.addColumn(questionnaireEditor.resources.GridColumnPrefix + '3', 'NumericAnswer');
            break;
        case GridEditor.starRatingGridType:
            this.addColumn(questionnaireEditor.resources.GridColumnPrefix + '1', 'StarRating');
            break;
        default:
            break;
    }

    this.addVariableRow(questionnaireEditor.resources.GridRowPrefix + '1');
    this.addVariableRow(questionnaireEditor.resources.GridRowPrefix + '2');
    this.addVariableRow(questionnaireEditor.resources.GridRowPrefix + '3');
};

GridEditor.prototype.getGridType = function () {
    if (this.Container)
        return this.Container.attr('data-grid-type') || GridEditor.hybridGridType;
    return null;
}

function fillElements(content, settings, typeSelector, inputClass) {
    _.each(settings, function (value, key) {
        var elements = content.find(typeSelector + ' ' + inputClass + '[data-name=' + key + ']');
        a4.setInputValue(elements, value);
    });
}

function saveElements(content, selector, saveto) {
    var newValues = {};

    $(selector, content).each(function () {
        var key = $(this).attr('data-name');
        var value = a4.getInputValue($(this));

        if (!(key in newValues) || value != "") {
            newValues[key] = value;
        }

        if ($(this).is(".openEndType:checkbox")) {
            newValues[key] = $(this).is(":checked") ? "1" : "0";
        }
    });

    saveto = _.extend(saveto, newValues);
}

function InitializeChoiceListEditorForColumn(editor, choiceListEditor, column) {
    // Choice
    $(".v-choiceListTable", choiceListEditor).attr("data-multipleselection", (column.ColumnType == "CheckBox" || column.ColumnType == "ImageSelector") ? "true" : "false");

    // 
    // Adding the disable columns header
    $('<th>').addClass('choice-disable-columns')
        .html(questionnaireEditor.resources.ChoiceListEditor_DisableColumns)
        .insertBefore($('thead th.choice-edition', choiceListEditor));

    // Adding the disable column cell
    $('<td>').addClass('choice-disable-columns')
        .append($('<select>').addClass('v-choiceDisableColumns')
            .attr('multiple', 'multiple')
            .append($.map(editor.getColumnSettings(),
                function (col, ix) { return (col && col.Position != column.Position) ? $('<option>').attr('value', ix).html(col.Label || '').get(0) : null; })))
        .insertBefore($('tbody td.choice-edition-column-buttons', choiceListEditor));

    _.each(column.Choices, function (choice) {
        var choiceRow = $(".v-newChoice tbody tr", choiceListEditor).clone(false);

        choiceRow.attr('id', "ChoiceRow" + choice.Id).attr('data-choice-list-id', column.ChoiceListId);

        if (choice.HideIf)
            choiceRow.attr('data-choice-skipif', JSON.stringify(choice.HideIf));
        else
            if (choice.SkipLogicJSON)
                choiceRow.attr('data-choice-skipif', choice.SkipLogicJSON);

        if (choice.DisplayIf)
            choiceRow.attr('data-choice-displayif', JSON.stringify(choice.DisplayIf));
        else
            if (choice.DisplayLogicJSON)
                choiceRow.attr('data-choice-displayif', choice.DisplayLogicJSON);


        var _hasLogic = (choice.HideIf || choice.DisplayIf || choice.SkipLogicJSON || choice.DisplayLogicJSON) != null;

        $(".h-choiceLogic", choiceRow).attr('data-haslogic', _hasLogic);
        $(".v-choiceCode", choiceRow).val(choice.Code);
        $(".v-choiceText", choiceRow).html(choice.Text).addClass('editable').addClass('v-qeRichEditText');
        $(".v-choiceOpenEndType", choiceRow).val(choice.OpenEndType);
        $(".v-defaultChoice:checkbox", choiceRow).prop('checked', choice.Default);
        $(".v-visibleChoice:checkbox", choiceRow).prop('checked', choice.Visible);
        $(".v-fixedChoice:checkbox", choiceRow).prop('checked', choice.Fixed);
        $(".v-exclusiveChoice:checkbox", choiceRow).prop('checked', choice.Exclusive);
        $(".v-choiceImageUrl", choiceRow).val(choice.TranslatableChoiceSettings && choice.TranslatableChoiceSettings.ImageUrl);
        $("select.v-choiceDisableColumns", choiceRow).val(_.map(choice.DisableColumns || null, function (_number) { return new Number(_number).toString(10); }));
        $(choiceRow).appendTo($("table .v-choiceListBody", choiceListEditor));

        // Create widgets after choiceRow is added to the document
        $(".v-choiceText", choiceRow).a4richTextEditor({ library: true, piping: true, inline: true });
        $("select.v-choiceDisableColumns", choiceRow).multiSelect();
    });

    choiceListEditor.on('rowAdded', function (args, row) {
        $(row).find('select.v-choiceDisableColumns').multiSelect();
    });

}

function InitializeGridColumnEditorValues(frame, column, editor) {
    var content = $(frame).contents();

    function canDisableColumns(type) {
        return _.include(['CheckBox', 'DropDownList', 'ImageSelector', 'RadioButton', 'Slider0'], type);
    }

    // Making sure there are TypeSettings
    if (!column.TypeSettings) { column.TypeSettings = {}; }
    if (!column.AxisTranslatableSettings) { column.AxisTranslatableSettings = {}; }

    var columnType = column.ColumnType;
    if (columnType == 'Slider') {
        columnType += column.TypeSettings.VarType || '0';
    }

    fillElements(content, column.TypeSettings, '.v-columnTypeSetting.type' + columnType, '.v-setting');
    fillElements(content, column.AxisTranslatableSettings, '.v-columnSetting', '.v-translatableSetting');
    fillElements(content, column.AxisTranslatableSettings, '.v-columnTypeSetting.type' + columnType, '.v-translatableSetting');
    fillElements(content, column, '.v-columnSetting', '.v-setting');

    if ($(".v-columnTypeSetting .v-selectMultipleVariables, .v-columnTypeSetting .v-EliminationExcludedFields, .v-columnTypeSetting .v-softPromptChoice", content).length > 0) {
        $(".v-columnTypeSetting .v-selectMultipleVariables, .v-columnTypeSetting .v-EliminationExcludedFields, .v-columnTypeSetting .v-softPromptChoice", content).each(function () {
            $(this).attr("data-value", $(this).val());
            $(this).val("");
        });
    }

    // Set Controls Behavior
    questionnaireEditor.updateDimensionSelectorValue(content.find('.v-dimensionSelectorContainer .v-setting'));

    $('.v-setting:input[type=checkbox][data-name=DisplayNoAnswer]', content).on("change", function () {
        $('.v-displayNoAnswer', content).toggle($(this).is(':checked'));
    });

    $('.v-setting:input[type=checkbox][data-name=DisplayTicks]', content).on("change", function () {
        var sliderType = $('.v-setting[data-name=VarType]:checked', content).val();
        $('.v-displayTicks', content).toggle($(this).is(':checked') && sliderType == 1);
    });

    choiceListEditor = $('.v-columnTypeSetting .v-choiceListEditor', content);
    InitializeChoiceListEditorForColumn(editor, choiceListEditor, column);

    $(".v-qeRichEditText:not(.v-choiceText)", content).a4richTextEditor({ library: true, piping: true, inline: true });

    // Updating the information displayed based on the column type
    var changedType = function () {
        var newColumnType = $('.v-setting[data-name=ColumnType]', content).val() || columnType;

        if (newColumnType === 'Slider') {
            newColumnType += $('.v-setting[data-name=VarType]:checked', content).val();
        }

        choiceListEditor.choiceEditor({
            DisplayTextColumn: newColumnType !== 'StarRating',
            DisplayDefaultColumn: true,
            DisplayExclusiveColumn: newColumnType == 'CheckBox',
            DisplayFixedColumn: false,
            DisplayImageColumn: _.include(['ImageSelector', 'Slider0', 'StarRating'], newColumnType),
            DisplayIndexColumn: false,
            DisplayOpenEndColumn: false,
            DisplayColorColumn: false,
            DisplayHeaderColumn: false,
            DisplayAvailableChoiceColumn: false,
            ShowChoiceListSelector: false
        });

        $('.choice-disable-columns', choiceListEditor).toggle(canDisableColumns(newColumnType));

        content.find('.v-columnTypeSetting').hide();
        content.find('.v-columnTypeSetting.type' + newColumnType).show();

        $('.v-toggleSetting', content).trigger("change");
    };

    content.find('.v-setting[data-name=ColumnType],.v-setting[data-name=VarType]').on("change", function () {
        changedType();
    });

    changedType();

    $(".v-minMaxValue", content).each(function () {
        questionnaireEditor.setMinMaxInitialValue($(this).closest(".v-minMaxValueContainer"));
    });

    questionnaireEditor.initializeNumericInputs(content);
    questionnaireEditor.initializeNumericAnswerInputs(content);

    $('.v-setting, .v-translatableSetting', content).trigger('change');

    var answerRequiredValue = $('.h-answerRequired', editor.Container.parents('.question-main-content')).val();

    $('.v-softPromptSection', content).toggle(answerRequiredValue == "SoftPrompt" || answerRequiredValue == "UniquePrompt");
}

function RetrieveGridColumnEditorValues(frame, _settings) {
    var content = $(frame).contents();

    if (!_settings) {
        _settings = new Object();
    }

    _settings.TypeSettings = {};
    _settings.AxisTranslatableSettings = {};

    var columnType = $('.v-setting[data-name=ColumnType]', content).val() || _settings.ColumnType;

    if (columnType == 'Slider') {
        columnType += $('.v-setting[data-name=VarType]:checked', content).val();
    }

    saveElements(content, '.v-columnSetting .v-setting', _settings);
    saveElements(content, '.v-columnTypeSetting.type' + columnType + ' .v-setting', _settings.TypeSettings);
    saveElements(content, '.v-columnSetting .v-translatableSetting', _settings.AxisTranslatableSettings);
    saveElements(content, '.v-columnTypeSetting.type' + columnType + ' .v-translatableSetting', _settings.AxisTranslatableSettings);

    // Choice
    if (columnType.toLowerCase() === 'text'
        || columnType.toLowerCase() === 'numericanswer'
        || columnType.toLowerCase() === 'slider1') {
        _settings.Choices = [];
    } else {
        _settings.Choices = _.map(content.find('.v-columnTypeSetting.type' + columnType + ' .v-choiceListTable .v-questionEditableItem:not(.deleted)'), function (_row) {
            var choice = questionnaireEditor.choiceObject($(_row), null, true);
            choice.DisableColumns = $(_row).find('select.v-choiceDisableColumns').val();
            return choice;
        });

        // Add at least one choice, with no label but a code
        if (!_.size(_settings.Choices) && columnType.toLowerCase() !== 'slider0') {
            _settings.Choices = [{ Code: 0, Text: '', Visible: true }];
        }
    }

    // Eliminate choices
    $(".v-columnTypeSetting .v-selectMultipleVariables", content).each(function () {
        _settings.TypeSettings[$(this).attr('data-name')] = _.pluck($(this).a4autocomplete("getValue"), "name").join(',');
    });

    $(".v-columnTypeSetting .v-EliminationExcludedFields", content).each(function () {
        _settings.TypeSettings[$(this).attr('data-name')] = _.pluck($(this).a4autocomplete("getValue"), "Code").join(',');
    });

    $(".v-columnTypeSetting .v-softPromptChoice", content).each(function () {
        var value = $(this).a4autocomplete("getValue");
        if (value) value = value.Code;
        _settings.TypeSettings[$(this).attr('data-name')] = value;
    });

    return _settings;
}

function InitializeGridColumnChoiceEditorValues(frame, column, choice, editor) {
    var content = $(frame).contents();

    //Prevents invalid characters in choice code
    $(".v-choiceCode", content).on("keypress", function (e) { return DisableInvalidCharacters(e); });

    if (!choice.OpenEnd) { choice.OpenEnd = { EnforceOpenEnd: true, MaxLength: null }; }
    if (!choice.OpenEnd.AnswerCode) { choice.OpenEnd.AnswerCode = 0; }
    if (!choice.OpenEnd.Lines) { choice.OpenEnd.Lines = 2; }

    // Set Open End
    var openEnd = $('.openEndType', content);

    if (!openEnd.is(":checkbox")) {
        var options = [];
        $.each(questionnaireEditor.openEndDisplayTypes, function (index, text) {
            if (index == 3) { text = questionnaireEditor.resources.OpenEndType_Column; }   // Change label display as row to column
            options.push($('<option>').attr('value', index).html(text));
        });
    }

    $.fn.append.apply(openEnd, options);

    var changeSettingVisibility = function (editContainer, openEndType) {
        $('.v-childOpenEndSetting', editContainer).toggle(openEndType != "0");
        $(".v-childOpenEndSetting.openEndTypeRow", editContainer).toggle(openEndType == "3");
        //TODO commented until we get feedback from client, do we allow writing bad values in this field.
        //$('.v-openEndMinLength', editContainer).autoNumeric({ wEmpty: 'zero', aPad: false, vMin: '0', vMax: '999999999999999', mDec: '0', aSep: '' });
    };

    changeSettingVisibility(content, choice.OpenEnd.Type || 0);
    openEnd.on("change", function () {
        var settings = $(this).closest(".v-openEndSettingsContainer");
        var value = $(this).val();
        if (value !== "0" && $('[data-name=VariableName]', settings).val() === "") {
            $('[data-name=VariableName]', settings).val(GetQuestionName(editor.Container).toUpperCase() + "_C" + (choice.Position + 1) + "_OE" + (column.Position + 1));
        }
        if ($(this).is(":checkbox")) {
            value = $(this).is(":checked") ? "1" : "0";
        }

        changeSettingVisibility(settings, value);
    });

    $('.v-setting.openEndHeight', content).attr("data-name", "Height");
    $('.v-setting.openEndWidth', content).attr("data-name", "Width");

    fillElements(content, choice.OpenEnd, '.v-columnChoiceOpenEndSetting', '.v-setting');
    fillElements(content, choice, '.v-columnChoiceSetting', '.v-setting');
    fillElements(content, choice.TranslatableChoiceSettings, '.v-columnChoiceCustomSetting', '.v-setting');

    $(".v-qeRichEditText", content).a4richTextEditor({ library: true, piping: true, inline: true });

    // HACK for DimensionSelector
    questionnaireEditor.updateDimensionSelectorValue(content.find('.v-dimensionSelectorContainer .v-setting'));

    if (column.ColumnType === 'CheckBox') {
        $('.v-setting[data-name=Exclusive]', content).parents('.v-columnChoiceSetting').show();
    } else {
        $('.v-setting[data-name=Exclusive]', content).parents('.v-columnChoiceSetting').hide();
    }

    if (column.ColumnType === 'ImageSelector' || column.ColumnType === 'Slider0') {
        $('.v-setting[data-name=ImageUrl]', content).parents('.v-columnChoiceSetting').show();
    } else {
        $('.v-setting[data-name=ImageUrl]', content).parents('.v-columnChoiceSetting').hide();
    }

    $('.v-setting', content).trigger('change');

    if (editor.getGridType() == GridEditor.netPromoterGridType) {
        $(".v-setting[data-name=Code]", content).addClass("readonly");
        $(".v-setting[data-name=Code]", content).attr("readonly", "readonly");
    }
}

function RetrieveGridColumnChoiceEditorValues(frame, column, choice) {
    var content = $(frame).contents();

    saveElements(content, '.v-columnChoiceSetting .v-setting', choice || {});

    choice.OpenEnd = choice.OpenEnd || {};
    saveElements(content, '.v-columnChoiceOpenEndSetting  .v-setting', choice.OpenEnd);

    choice.TranslatableChoiceSettings = choice.TranslatableChoiceSettings || {};
    saveElements(content, '.v-columnChoiceCustomSetting .v-setting', choice.TranslatableChoiceSettings);
}

///
/// Initialization of the HybridGrid
///
///
/// Popup windows
///

var $gridEditorOptions = '';
$.fn.gridEditor = function (_options) {

    $gridEditorOptions = _options;
    var returnValue = this;

    this.each(function () {
        var _self = $(this);
        var _editor = new GridEditor(this);

        function createCell(_content) {
            return $('<td>').append(_content);
        }

        function createHeaderCell(_content) {
            return $('<th>').append(_content);
        }

        function createEditableTextCell(_content, _spanClasses, _emptyMessage, _createCellCallback) {
            var _cell = (_createCellCallback || createCell)($('<div>').attr('id', _.uniqueId('gridText')).addClass(_spanClasses).append(_content));
            _cell.addClass('editableCell');
            return _cell;
        }

        function createRichEditableTextCell(_content, _spanClasses, _emptyMessage, _createCellCallback, _hideIf) {
            var _cell = (_createCellCallback || createCell)($('<div>').attr('id', _.uniqueId('gridText')).addClass(_spanClasses));
            var _span = _cell.find('div');
            _span.addClass('editable v-qeRichEditText v-questionEditText rich-text-editor');
            _span.html(_content);
            _cell.addClass('editableCell');

            var _class = _spanClasses.indexOf("gridColumn") >= 0 ? "v-linkEditColumnLogic" : "v-linkEditRowLogic";
            if (_hideIf && (_hideIf.CustomCondition.length > 0 || _hideIf.Conditions.length > 0))
                _cell.prepend("<div><span class='" + _class + " fa fa-eye'  title='" + questionnaireEditor.resources.EditColumnLogic + "'></span></div>");

            return _cell;
        }

        function getGridType() {
            if (_self instanceof jQuery)
                return _self.attr('data-grid-type') || GridEditor.hybridGridType;
            return null;
        }

        function isChoiceGridType() {
            var type = getGridType();
            return type == GridEditor.choiceGridType || type == GridEditor.imageChoiceGridType || type == GridEditor.netPromoterGridType;
        }

        function isNetPromoter() {
            var type = getGridType();
            return type == GridEditor.netPromoterGridType;
        }

        function isSingleColumnGridType() {
            var type = getGridType();
            return isChoiceGridType() || type == GridEditor.starRatingGridType; // In case we have different type of grid than choice grids that have a single column.
        }

        function isSingleDisplayedColumnGridType() {
            var type = getGridType();
            return type == GridEditor.starRatingGridType; // In case we have different type of grid than choice grids that have a single column.
        }

        function gridCanHaveChoiceColumns() {
            return isChoiceGridType() || getGridType() === GridEditor.hybridGridType;
        }

        function getColumnSelector(_type, _column) {
            return _type + '[data-column' + (_column ? ('=' + $(_column).attr('data-column')) : '') + ']';
        }

        function getColumnChoiceSelector(_type, _columnChoice) {
            return _type + '[data-column' + (_columnChoice ? ('=' + $(_columnChoice).attr('data-column')) : '') + ']'
                + '[data-choice-index' + (_columnChoice ? ('=' + $(_columnChoice).attr('data-choice-index')) : '') + ']';
        }

        function createVariableCell(_row, _column, _columnIndex) {
            var cellContent = createEditableTextCell(_row.VariableNames[_columnIndex], 'gridVariable', questionnaireEditor.resources.EditText).attr('data-column', _columnIndex);

            if (!isSingleColumnGridType() && _columnIndex != null) {
                if (_row.VariableHideIfs != undefined
                    && _row.VariableHideIfs.length > 0
                    && _columnIndex < _row.VariableHideIfs.length
                    && _row.VariableHideIfs[_columnIndex]
                    && (_row.VariableHideIfs[_columnIndex].CustomCondition.trim().length > 0
                        || _row.VariableHideIfs[_columnIndex].Conditions.length > 0)
                ) {
                    cellContent.prepend($('<span>').attr('class', 'fa fa-eye v-linkEditCellLogic').attr('title', questionnaireEditor.resources.EditCellLogic));
                }
                else {
                    cellContent.prepend($('<span>').attr('class', 'fa fa-eye skip v-linkEditCellLogic').attr('title', questionnaireEditor.resources.EditCellLogic));
                }
            }
            return cellContent;
        }

        function createControlCells(_row, _column, _columnIndex) {
            var isChoiceCell = (_column.ColumnType == 'RadioButton' || _column.ColumnType == 'CheckBox' || _column.ColumnType == 'ImageSelector') && _.size(_column.Choices) > 0;

            if (_row.RowType == GridEditor.textRowType && !_row.MergeCells && _row.UseChoiceTexts) {
                return createChoiceHeaderCells(_column, _columnIndex);
            } else if (_row.RowType == GridEditor.variableRowType) {

                var _name = _.uniqueId('input');

                function getInputText(_choiceIndex) {
                    var input = "";
                    switch (_column.ColumnType) {
                        case 'RadioButton': input += '<input name="' + _name + '" type="radio" />'; break;
                        case 'CheckBox': input += '<input name="' + _name + '" type="checkbox" />'; break;
                        case 'ImageSelector': input += '<img src="' + ((_column.Choices[_choiceIndex].TranslatableChoiceSettings && _column.Choices[_choiceIndex].TranslatableChoiceSettings.ImageUrl) || '') + '" style="width:90%; height:90%; alt="" />'; break;
                        case 'NumericAnswer':
                        case 'Text': input += '<input name="' + _name + '" type="text" />'; break;
                        case 'Slider': input += '<div class="sliderPreview" data-choice-count="' + (_.size(_column.Choices) || 1) + '" />'; break;
                        case 'DropDownList': input += '<select>' + _.map(_column.Choices, function (_choice) { return '<option>' + a4.decodeHtml(_choice.Text) + '</option>'; }).join('') + '</select>'; break;
                        case 'StarRating': input += '<img src="../../../Content/Images/Icons/StarRatingPreview.png" />'; break;
                        default: input = null; break;
                    }

                    return input;
                }


                return $($.map(_.range(_editor.numCells(_column)), function (_value, _index) {
                    var _cell = $('<td class="previewCell">' + (getInputText(_index) || '') + '</td>').attr('data-column', _columnIndex);
                    if (isChoiceCell) {
                        _cell.attr('data-choice-index', _index);
                    }
                    return _cell.get(0);
                }));
            }

            return null;
        }

        function createDataCell(_row, _column, _columnIndex) {
            return _table.hasClass('previewMode') ? createControlCells(_row, _column, _columnIndex) : createVariableCell(_row, _column, _columnIndex);
        }

        function createChoiceHeaderCells(_column, _columnIndex) {
            if (_table.hasClass('previewMode')) {
                if (_column.ColumnType == 'RadioButton' || _column.ColumnType == 'CheckBox' || _column.ColumnType == 'ImageSelector') {
                    return $($.map(_column.Choices, function (_choice, _index) {
                        return createRichEditableTextCell(_choice.Text, 'gridColumnChoiceLabel', $('.emptyEditMessage').text(), createHeaderCell, _choice.HideIf)
                            .attr('data-choice-index', _index).get(0);
                    })).attr('data-column', _columnIndex);
                }
            }

            return null;
        }

        function createMoveHandleCell() {
            return $('<td>').addClass('rowGripColumn').append($('<span>').addClass('fa fa-lg fa-ellipsis-v verticalGridMoveHandle'));
        }

        function createLeftLabelCell(row, _isPreviewMode) {
            return createRichEditableTextCell(row.LeftLabel, 'gridLeftLabel', $('.emptyEditMessage').text(), createCell, _isPreviewMode ? row.HideIf : "").addClass('columnLeftLabel');
        }

        function createRightLabelCell(label) {
            return createRichEditableTextCell(label, 'gridRightLabel', $('.emptyEditMessage').text()).addClass('columnRightLabel');
        }

        function createColumnContentCell(label, hideIf) {
            return createRichEditableTextCell(label, 'gridColumnLabel', $('.emptyEditMessage').text(), createHeaderCell).addClass('contentLabel');
        }

        function createRow(editor, row, index) {

            if (_table.hasClass('previewMode') || row.RowType == GridEditor.variableRowType) {

                var tr = $('<tr class="editableRow">');
                tr.attr('data-row', index);

                if (row.RowType == GridEditor.variableRowType) {
                    tr.addClass('variableRow');
                } else if (row.RowType == GridEditor.titleRowType) {
                    tr.addClass('titleRow');
                } else if (row.RowType == GridEditor.textRowType) {
                    tr.addClass('textRow');
                }

                tr.append(createMoveHandleCell());

                if (!((row.RowType == GridEditor.titleRowType || row.RowType == GridEditor.textRowType) && row.MergeCells && row.MergeLeftLabel)) {
                    tr.append(createLeftLabelCell(row, _table.hasClass('previewMode')));
                }

                var tmpIndexes = [];
                _.each(editor.getColumnSettings(), function (column, ix) { if (column) { tmpIndexes[column.Position] = ix; } });

                if (row.RowType == GridEditor.titleRowType || row.RowType == GridEditor.textRowType) {
                    if (row.MergeCells) {
                        var colSpan = editor.numCells() + (row.MergeLeftLabel || 0) + (row.MergeRightLabel || 0);
                        var contentCell;
                        if (row.RowType == GridEditor.textRowType) {
                            contentCell = createColumnContentCell((row.Labels && row.Labels[0] && row.Labels[0][0]) || '')
                                .attr('data-column', "0")
                                .attr('data-choice-index', "0")
                                .attr('colspan', colSpan);
                        } else {
                            contentCell = createColumnContentCell((row.Labels && row.Labels[0]) || '')
                                .attr('data-column', "0")
                                .attr('colspan', colSpan);
                        }

                        tr.append(contentCell);
                    } else {
                        _.each(editor.getOrderedColumns(), function (column) {
                            var columnIndex = tmpIndexes[column.Position];
                            var numColumnCells = editor.numCells(column);
                            if (row.RowType == GridEditor.textRowType) {
                                var cells = createControlCells(row, column, columnIndex);
                                if (!cells) {
                                    cells = $($.map(_.range(numColumnCells), function (value, index) {
                                        var cell = createColumnContentCell((row.Labels && row.Labels[column.Position] && row.Labels[column.Position][index]) || '', column.HideIf)
                                            .attr('data-column', columnIndex)
                                            .attr('data-choice-index', index);
                                        return cell.get(0);
                                    }));
                                }
                                tr.append(cells);
                            } else {
                                var _hideIf = row.Position == 0 ? column.HideIf : "";
                                tr.append(createRichEditableTextCell(
                                    (row.Labels && row.Labels[column.Position]) || '',
                                    'gridColumnLabel',
                                    $('.emptyEditMessage').text(),
                                    createHeaderCell,
                                    _hideIf)
                                    .attr('data-column', columnIndex)
                                    .attr('colspan', numColumnCells));
                            }
                        });
                    }
                } else {
                    _.each(_.sortBy(_.compact(editor.getColumnSettings()), function (column) { return column.Position; }),
                        function (column) {
                            var cells = createDataCell(row, column, tmpIndexes[column.Position]);
                            tr.append(cells);
                            $('.sliderPreview', cells).each(function () {
                                $(this).slider({ max: $(this).attr('data-choice-count') });
                            });
                        });
                }



                if (!((row.RowType == GridEditor.titleRowType || row.RowType == GridEditor.textRowType) && row.MergeCells && row.MergeRightLabel)) {
                    tr.append(createRightLabelCell(row.RightLabel));
                }

                tr.append('<td class="columnDeleteRow">'); // Cell for delete row icon - will be created on-demand when hovering
                return tr;
            } else {
                return null;
            }

        }

        function addRowInTable(editor, row, index, _isVariableMode) {
            var tr = createRow(editor, row, index);
            if (tr) {
                var Rowlength = $("span.fa.fa-eye.v-linkEditCellLogic", tr).length;
                var _type = $(editor.Container).attr("data-grid-type");
                if ((row.Position == 0 || (_isVariableMode && Rowlength > 0)) && _type != "StarRatingGrid" && (row.RowType == 1 || row.RowType == 3))//
                    createHideIfRow(tr, _isVariableMode);

                $('tbody:first', _table).append(tr);  // Add row only inside body of table (prevent from inserting in child tables)
                $(".v-qeRichEditText", tr).a4richTextEditor({ library: true, piping: true, inline: true }); // Create a4richTextEditor after tr is added to the document
            }
        }

        function addColumnOperationRow(editor) {

            if (isSingleDisplayedColumnGridType()) {
                return null;
            } else {
                var tr = $('<tr>').addClass('v-columnOperations');

                tr.append($('<td>')); // row grip
                tr.append($('<td>')); // left label

                if (isChoiceGridType()) {
                    _.each(_.first(editor.getOrderedColumns()).Choices, function (choice, index) {
                        var td = $('<td>').attr('data-column', 0)
                            .attr('data-choice-index', index);
                        tr.append(td);
                    });
                } else {
                    var tmpIndexes = [];
                    _.each(editor.getColumnSettings(), function (column, ix) { if (column) { tmpIndexes[column.Position] = ix; } });

                    _.each(editor.getOrderedColumns(), function (column) {
                        var td = $('<td>').attr('data-column', tmpIndexes[column.Position]);
                        var numCells = editor.numCells(column);
                        if (numCells > 1) {
                            td.attr('colspan', numCells);
                        }

                        tr.append(td);
                    });
                }

                tr.append($('<td>')); // right label
                tr.append($('<td>')); // delete row

                return tr.appendTo($('thead', _table));
            }

        }

        function addColumnInTable(_editor, _column, _index) {
            var _colgroup = $('<colgroup class="editableColumn" data-column="' + _index + '" data-coltype="' + _column.ColumnType + '" />').insertBefore($('colgroup.columnRightLabel', _table));

            if (_table.hasClass('previewMode')) {
                _.times(_editor.numCells(_column), function () { _colgroup.append($('<col />')); });
            }

            // Adding footer row cell
            $('<td>').attr('data-column', _index)
                .attr('colspan', _table.hasClass('previewMode') ? _editor.numCells(_column) : 1)
                .insertBefore(_table.find('tfoot tr td.columnRightLabel'));
        }

        function renderTableStructure() {
            CloseRowEdition();

            if (_self instanceof jQuery) {
                // Inserting new table
                $('.ui-grid-container table.ui-grid-editor', _self).remove();
                _table = $('<table class="ui-grid-editor"></table>');
                $('.ui-grid-container', _self).append(_table);

                _self.find('.grid-mode .previewMode').removeClass('currentMode').on('click.render', renderPreviewModeTable);
                _self.find('.grid-mode .variableMode').removeClass('currentMode').on('click.render', renderVariableModeTable);

                _table.append('<colgroup class="moveHandle"><col /></colgroup>')       // Enable row movement by dragging
                    .append('<colgroup class="columnLeftLabel"><col /></colgroup>')  // Left label
                    .append('<colgroup class="columnRightLabel"><col /></colgroup>') // Right label
                    .append('<colgroup class="columnDeleteRow"><col /></colgroup>')  // Column for delete row icon
                    .append(getGridType() === GridEditor.runningTotalGridType
                        ? $('<tfoot>').append($('<tr>').append('<td>')
                            .append(createRichEditableTextCell($('.v-translatableSetting[data-name=TotalRowLabel]', _self).val(), 'gridFooterLeftLabel', $('.emptyEditMessage').text(), createCell))
                            .append('<td class="columnRightLabel">')
                            .append('<td class="columnDeleteRow">'))
                        : '')
                    .append('<thead>')
                    .append('<tbody>');
            }
        }

        function renderPreviewModeTable() {
            renderTableStructure();
            _table.addClass('previewMode');
            _self.find('.grid-mode .previewMode').addClass('currentMode').off('click.render');
            _self.find(".v-renameVariables").addClass("hidden");

            var tmpIndexes = [];
            _.each(_editor.getColumnSettings(), function (column, ix) { if (column) { tmpIndexes[column.Position] = ix; } });
            _.each(_.sortBy(_.compact(_editor.getColumnSettings()), function (column) { return column.Position; }),
                function (column) { addColumnInTable(_editor, column, tmpIndexes[column.Position]); });

            addColumnOperationRow(_editor);

            tmpIndexes = [];
            _.each(_editor.getRowSettings(), function (row, ix) { if (row) { tmpIndexes[row.Position] = ix; } });
            _.each(_.sortBy(_.compact(_editor.getRowSettings()), function (row) { return row.Position; }),
                function (row, ix) { addRowInTable(_editor, row, tmpIndexes[row.Position], false); });

            $(".v-qeRichEditText", _self).a4richTextEditor({ library: true, piping: true, inline: true });

        }

        function renderVariableModeTable() {
            renderTableStructure();
            if (_table) {
                _table.addClass('variableMode');
                if (_self instanceof jQuery) {
                    _self.find('.grid-mode .variableMode').addClass('currentMode').off('click.render');
                    _self.find(".v-renameVariables").removeClass("hidden");

                    var _tmpIndexes = [];
                    _.each(_editor.getColumnSettings(), function (_column, _ix) { if (_column) { _tmpIndexes[_column.Position] = _ix; } });
                    _.each(_.sortBy(_.compact(_editor.getColumnSettings()), function (_column) { return _column.Position; }),
                        function (_column) { addColumnInTable(_editor, _column, _tmpIndexes[_column.Position]); });

                    _tmpIndexes = [];
                    _.each(_editor.getRowSettings(), function (_row, _ix) { if (_row) { _tmpIndexes[_row.Position] = _ix; } });
                    _.each(_.sortBy(_.compact(_editor.getRowSettings()), function (_row) { return _row.Position; }),
                        function (_row, _ix) { addRowInTable(_editor, _row, _tmpIndexes[_row.Position], true); });

                    $(".gridVariable", _self).a4richTextEditor({ fullToolbar: false, library: false, piping: false, inline: true, multiLine: false });
                }
            }
        }

        function refreshRow(tr) {
            var _position = $(tr).attr('data-row');
            var newRow = createRow(_editor, _editor.getRowSettings(tr), _position);
            var _type = $(_editor.Container).attr("data-grid-type");
            if (_position == 0 && $("span.fa-eye", newRow).length > 0 && _type != "StarRatingGrid" && ($(newRow).hasClass('titleRow') || $(newRow).hasClass('textRow')))
                newRow = createHideIfRow(newRow, false);

            $(tr).replaceWith(newRow);
            $(".v-qeRichEditText", newRow).a4richTextEditor({ library: true, piping: true, inline: true }); // Create a4richTextEditor after newRow is added to the document
            return newRow;
        }

        function refreshTable() {
            if (_table == undefined) {
                _table = $(".gridEditor table.ui-grid-editor", questionnaireEditor.getInEditQuestion());
            }

            var rowInEdition = _editor.getRowIndex($('tr.editing', _table));
            CloseRowEdition();

            if (_table.hasClass('variableMode')) {
                renderVariableModeTable();
            }
            else {
                renderPreviewModeTable();
                if (rowInEdition) {
                    OpenRowEditor($('[data-row=' + rowInEdition + ']', _table));
                }
            }
        }

        function refreshClientAutoCompletes() {
            if (isChoiceGridType()) {
                var column = _.first(_editor.getColumnSettings() || []);
                $(".v-selectChoiceGridChoice", _self).a4autocomplete("option", "data", column.Choices);
            }
        }

        function createHideIfRow(_tr, _isVariableMode) {
            var clonedTr = _tr.clone();

            clonedTr.attr("class", "");
            clonedTr.attr("data-row", "-" + _tr.attr("data-row"));

            $("div.editable", clonedTr).remove();
            $("span.verticalGridMoveHandle", clonedTr).remove();



            if (_isVariableMode) {
                $('tbody:first', _table).append(clonedTr);
                $("div.gridVariable", clonedTr).remove();
            }
            else {
                $("td,th", clonedTr).attr("class", "");
                $('span.v-linkEditColumnLogic', clonedTr).each(function (index, item) {

                    var _cell = $(item).closest('th,td');
                    var _dataColumn = _cell.attr('data-column');

                    var _dataCoiceIndex = _cell.attr('data-choice-index');
                    var _dataCoiceIndexAttr = (_dataCoiceIndex != null ? '[data-choice-index="' + _dataCoiceIndex + '"]' : '');

                    var _cellToAddIn = $('tr.v-columnOperations [data-column="' + _dataColumn + '"]' + _dataCoiceIndexAttr + '');

                    if (_cellToAddIn.length > 0) {
                        var div = $(_cellToAddIn).find('div.showOnHover');

                        if (div.length === 0) {
                            div = $('<div class="showOnHover">')
                                .append('<span class="column-logic-edit v-linkEditColumnLogic showAllTime" title="' + questionnaireEditor.resources.EditColumnLogic + '"   ><i class="fa fa-eye" style="font-size: 15px;"></i></span>')
                                .css({ 'display': 'block', 'z-index': 0, 'text-align': 'center', 'position': 'relative', 'min-width': '70px' });

                            $(_cellToAddIn).append(div);

                        }
                        var _spans = $("span:not(.showAllTime)", div);
                        div.stop(true, true).show(); // Needed for the position call to work
                        div.position({
                            of: _cellToAddIn,
                            collision: 'none'
                        });

                        _spans.stop(true, true).fadeIn();
                    }
                })

            }

            $("span.v-linkEditColumnLogic.fa.fa-eye", _tr).remove();

            return _tr;
        }

        /////////////////////////////////////////////////////////////////
        // Setup the Add Rows & Columns button
        /////////////////////////////////////////////////////////////////
        $('.v-addRowsAndColumn .v-addRowsAndColumnsText:not(:empty)', _self).html(isSingleDisplayedColumnGridType() ? questionnaireEditor.resources.Button_AddEditRows : questionnaireEditor.resources.Button_AddEditRowsAndColumns);

        _self.find('.v-addRowsAndColumn span').on("click", function () {
            var originalRowLabels = _.pluck(_.sortBy(_.filter(_.compact(_editor.getRowSettings()), function (row) { return row.RowType == GridEditor.variableRowType }), function (row) { return row.Position; }), "LeftLabel");

            var originalColumnLabels;

            if (isChoiceGridType()) {
                originalColumnLabels = _.pluck(_.first(_editor.getColumnSettings()).Choices, "Text");
            }
            else {
                originalColumnLabels = _.map(_.compact(_editor.getColumnSettings()), function (c) { return (c.Label || "").trim().length > 0 ? c.Label : questionnaireEditor.resources.EmptyPipingText });
            }

            var editContent = $("#addEditMultipleForGrids");
            $("#columnDivision", editContent).toggle(!isSingleDisplayedColumnGridType());
            $(".columnValues", editContent).empty();
            $(".rowValues", editContent).empty();
            $(".columnValues", editContent).append(originalColumnLabels.join("\r\n"));
            $(".rowValues", editContent).append(originalRowLabels.join("\r\n"));

            var content = $('<div id="AddMultipleDialogContainer">').append(editContent.html());

            var addMultipleResources = {
                "Title": questionnaireEditor.resources.AddEditRowsAndColumns,
                "Message": content
            };

            modalDialog.showConfirmDialog(addMultipleResources,
                function () {
                    var text = questionnaireEditor.resources.EmptyPipingText.replace('[', '\\[').replace(']', '\\]');
                    var re = new RegExp(text, "g");
                    var columnLabels = ($(".columnValues", content).val() || '').replace(re, '').replace('\[\]', '');
                    columnLabels = columnLabels.split('\n');
                    var rowLabels = $(".rowValues", content).val().split('\n');

                    if (isSingleDisplayedColumnGridType()) {
                        columnLabels = originalColumnLabels;
                    }

                    if (editMultipleRowsColumnsTranslationDesync(originalColumnLabels, columnLabels, originalRowLabels, rowLabels)) {
                        var editor = _self.closest("#gridEditor");
                        $(".v-desyncTitle", editor).show();
                    }

                    SaveRowsAndColumns(columnLabels, rowLabels);
                }, 400, 600);

        });

        _self.find('.v-renameVariables span').on("click", function () {
            var content = $('<div id="renameVariablesDialogContainer">').append($("#renameVariablesForGrids").html());
            $(".renameByTextValue", content).keypress(function (e) {
                $("input[type=radio]", $(this).parent()).prop("checked", true);
                return DisableInvalidCharacters(e);
            });
            var renameVariablesResources = {
                "Title": questionnaireEditor.resources.RenameVariables,
                "Message": content
            };

            modalDialog.showConfirmDialog(renameVariablesResources,
                function () {
                    var editor = _self.closest("#gridEditor");

                    var renameType = $("input[name=autoRenameVariables]:checked", content).val();
                    var namePrefix = "";
                    if (renameType == 0) namePrefix = $(".v-questionName", editor.parents(".editor-content").first()).val();
                    else namePrefix = $(".renameByTextValue", content).val();

                    var namingOptions = $("input[name=namingOptions]:checked", content).val();
                    var renameFunction = function (col, colIndex, rowIndex) {
                        var colIndexName = (colIndex + 1);
                        if (namingOptions == 1) return namePrefix + "_R" + rowIndex + "_C" + colIndexName;
                        else if (namingOptions == 2) return namePrefix + "_C" + colIndexName + "_R" + rowIndex;
                        else return namePrefix + "_" + _editor.generateColumnCode(col, colIndex) + rowIndex;
                    };

                    var variableEditor = $(".variableMode", editor);
                    var rowIndex = 0, columnIndex = 0;
                    $(".editableRow.variableRow", variableEditor).each(function () {
                        rowIndex++;

                        var rowElement = $(this);
                        var row = _editor.getRowSettings(rowElement);

                        columnIndex = 0;

                        $(".gridVariable", $(this)).each(function () {
                            var $t = $(this);

                            var colElement = $t.parents('td')[0];
                            var column = _editor.getColumnSettings(colElement);

                            var newName = renameFunction(column, columnIndex, rowIndex);

                            if ($t.is("input"))
                                $t.val(newName);
                            else if ($t.is(".rich-text-editor"))
                                $t.a4richTextEditor("setContent", newName);
                            else
                                $t.html(newName);

                            $t.data("value", newName);

                            var variableColumnIndex = parseInt($(colElement).attr("data-column"), 10);

                            if (!isNaN(variableColumnIndex) && variableColumnIndex >= 0) {
                                row.VariableNames[variableColumnIndex] = newName;
                            }

                            columnIndex++;
                        });

                        _editor.setRowSettings(row, rowElement);
                    });
                }, 380, 600, false, null, {
                validateFunction: function () {
                    var editor = _self.closest("#gridEditor");
                    var renameType = $("input[name=autoRenameVariables]:checked", content).val();
                    if (renameType == 1) {
                        var namePrefix = $(".renameByTextValue", content).val();
                        var validCharacterRegex = /^[a-zA-Z_]+[0-9a-zA-Z_]*$/;

                        if (namePrefix.length == 0 || !validCharacterRegex.test(namePrefix)) {
                            a4.showErrorMessage(questionnaireEditor.resources.ErrorVariableInvalidSyntax.replace("{0}", namePrefix));
                            return false;
                        }
                        return true;
                    }

                    return true;
                }
            });


        });

        function editMultipleRowsColumnsTranslationDesync(originalColumnLabels, columnLabels, originalRowLabels, rowLabels) {
            var hasDesync = questionnaireEditor.languages.length > 1 && (originalColumnLabels.length > columnLabels.length || originalRowLabels.length > rowLabels.length);

            if (!hasDesync && questionnaireEditor.languages.length > 1) {
                for (var i = 0; !hasDesync && i != originalColumnLabels.length; i++) {
                    hasDesync = originalColumnLabels[i] != columnLabels[i];
                }

                if (!hasDesync) {
                    for (var j = 0; !hasDesync && j != originalRowLabels.length; j++) {
                        hasDesync = originalRowLabels[j] != rowLabels[j];
                    }
                }
            }

            return hasDesync;
        }

        function SaveRowsAndColumns(columnLabels, rowLabels) {
            var titleRowLabels = new Array();

            //Adding/Editing Columns
            if (isChoiceGridType()) {
                var choices = new Array();
                var column = _.first(_editor.getColumnSettings() || []);
                var choicePos = 0;

                if (!column) {
                    column = _editor.addColumn('');
                    if (getGridType() === GridEditor.imageChoiceGridType) {
                        column.ColumnType = 'ImageSelector';
                    } else {
                        column.ColumnType = $('.choice-grid-answer-type input:checked', _self).val() || 'RadioButton';
                    }
                }

                if (!_.isArray(column.Choices)) {
                    column.Choices = [];
                }
                else {
                    _.each(column.Choices, function (choice) {
                        if (columnLabels.length > 0) {
                            var choiceText = a4.encodeHtml(columnLabels.shift().trim());

                            titleRowLabels[choicePos] = choiceText;
                            choice.Text = choiceText;
                            choice.Position = choicePos;
                            choicePos++;
                            choices.push(choice);
                        }
                    });
                }

                //Add additional columns
                _.each(_.map(columnLabels, function (label) { return a4.encodeHtml(label.trim()); }), function (label) {
                    choices.push({ Code: (choicePos + (isNetPromoter() ? 0 : 1)).toString(), Text: label, Position: choicePos, Visible: true });
                    choicePos++;
                });

                column.Choices = choices;

                _editor.setColumnSettings([column]);

                refreshClientAutoCompletes();
            }
            else {
                var columns = new Array();
                var columnPos = 0;

                _.each(_editor.getColumnSettings(), function (column) {
                    if (columnLabels.length > 0) {
                        if (column) {
                            var columnLabel = a4.encodeHtml(columnLabels.shift().trim());

                            if (columnLabel != "") {
                                titleRowLabels[columnPos] = columnLabel;
                                column.Label = columnLabel;
                                column.Position = columnPos;
                                columnPos++;
                            }
                        }

                        columns.push(column);
                    }
                    else {
                        columns.push(null);

                        // Remove variables of column in rows
                        var rows = _.map(_editor.getRowSettings(), function (row) {
                            if (row && row.VariableNames) {
                                row.VariableNames[columnPos] = null;
                            }

                            if (row && row.VariableHideIfs) {
                                row.VariableHideIfs[columnPos] = null;
                            }

                            return row;
                        });
                        columnPos++;
                        _editor.setRowSettings(rows);
                    }
                });


                _editor.setColumnSettings(columns);

                _.each(_.map(columnLabels, function (columnLabel) { return a4.encodeHtml(columnLabel.trim()); }), function (columnLabel) {
                    titleRowLabels[columnPos] = columnLabel;

                    if (getGridType() === GridEditor.runningTotalGridType) {
                        _editor.addColumn(columnLabel, 'NumericAnswer');
                    }
                    else {
                        _editor.addColumn(columnLabel);
                    }

                    columnPos++;
                });
            }

            //Adding/Editing Rows        


            /********
            *     Complex Logic for replacing Variable Hide Ifs - We do not have the information on the original labels or the changes that were made, only the new labels
            *                                                      We change the row owner of the variableHideIfs depending on label changes and label count changed
            */

            var rows = new Array();
            var prevGridRowCount = _.size(_.filter(_editor.getRowSettings(), function (rs) { return rs && rs.RowType == GridEditor.variableRowType; }));
            var nbLabels = rowLabels.length;
            var lastRowVariableHideIfs = [];
            var previousRow = null;
            var skippedFirstDeletedRow = false;
            var encounteredFirstRowlabelChange = false;

            _.each(_editor.getRowSettings(), function (row) {
                if (rowLabels.length > 0) {
                    if (row) {
                        if (row.RowType == GridEditor.variableRowType) {
                            var rowLabel = a4.encodeHtml(rowLabels.shift().trim());
                            var temp = row.VariableHideIfs;

                            if (prevGridRowCount != nbLabels && (row.LeftLabel != rowLabel || encounteredFirstRowlabelChange)) {
                                if (nbLabels > prevGridRowCount) {
                                    row.VariableHideIfs = lastRowVariableHideIfs;
                                }
                                else if (nbLabels < prevGridRowCount) {
                                    if (previousRow != null && skippedFirstDeletedRow) {
                                        previousRow.VariableHideIfs = temp;
                                    }

                                    skippedFirstDeletedRow = true;
                                }

                                encounteredFirstRowlabelChange = true;
                            }

                            if (rowLabel != "") {
                                row.LeftLabel = rowLabel;
                            }

                            lastRowVariableHideIfs = temp;
                        }
                    }

                    rows.push(row);
                }
                else {
                    if (nbLabels < prevGridRowCount && previousRow != null && row) {
                        previousRow.VariableHideIfs = row.VariableHideIfs;
                    }

                    rows.push(null);
                }

                previousRow = row;
            });

            if (isChoiceGridType()) {
                var firstTextRow = _.first(_.filter(_.sortBy(_.compact(rows), function (row) { return row.Position; }), function (row) { return row.RowType == GridEditor.textRowType; }));
                if (firstTextRow && !firstTextRow.UseChoiceTexts) {
                    firstTextRow.Labels[0] = titleRowLabels;
                }
            }
            else {
                var firstTitleRow = _.first(_.filter(_.sortBy(_.compact(rows), function (row) { return row.Position; }), function (row) { return row.RowType == GridEditor.titleRowType; }));
                if (firstTitleRow) {
                    firstTitleRow.Labels = titleRowLabels;
                }
            }

            _editor.setRowSettings(rows);

            _.each(_.map(rowLabels, function (label) { return a4.encodeHtml(label.trim()); }), function (label) {
                var newRow = _editor.addVariableRow(label, lastRowVariableHideIfs);
                lastRowVariableHideIfs = [];    // Only the remaining last row can be saved (previous line of code)                
            });

            refreshTable();
        }

        // Header update --- Keeping this one only for the DisableIf Options
        _self.off('change', '.titleRow:first th div.gridColumnLabel').on('change', '.titleRow:first th div.gridColumnLabel', function () {
            var column = $(this).parents('th')[0];
            var colSetting = _editor.getColumnSettings(column);
            colSetting.Label = $(this).a4richTextEditor("getContent");
            _editor.setColumnSettings(colSetting, column);
        });

        // Choice Header update
        _self.off('change', '[data-column] div.gridColumnChoiceLabel').on('change', '[data-column] div.gridColumnChoiceLabel', function () {
            var cell = $(this).parents('th');
            var column = _editor.getColumnSettings(cell);
            var choiceIndex = cell.attr('data-choice-index');
            var content = $(this).a4richTextEditor("getContent");
            column.Choices[choiceIndex].Text = content;
            _editor.setColumnSettings(column, cell);

            // Refreshes all choice cells
            $('[data-column=' + _editor.getColumnIndex(cell) + '][data-choice-index=' + choiceIndex + '] .gridColumnChoiceLabel', _self).not($(this)).a4richTextEditor("setContent", content);
        });

        //Left label update
        _self.off('change', 'tr div.gridLeftLabel').on('change', 'tr div.gridLeftLabel', function () {
            var _row = $(this).parents('tr')[0];
            var _rowSetting = _editor.getRowSettings(_row);
            _rowSetting.LeftLabel = $(this).a4richTextEditor("getContent");
            _editor.setRowSettings(_rowSetting, _row);
        });

        // Footer Left Label
        _self.off('change', 'td div.gridFooterLeftLabel').on('change', 'td div.gridFooterLeftLabel', function () {
            $(".v-translatableSetting[data-name=TotalRowLabel]", _self).val($(this).a4richTextEditor("getContent"));
        });

        // Right label update
        _self.off('change', 'tr div.gridRightLabel').on('change', 'tr div.gridRightLabel', function () {
            var _row = $(this).parents('tr')[0];
            var _rowSetting = _editor.getRowSettings(_row);
            _rowSetting.RightLabel = $(this).a4richTextEditor("getContent");
            _editor.setRowSettings(_rowSetting, _row);
        });

        // Open end Row label update
        _self.off('change contentChange', 'tr div.openEndRowSettings div.openEndTypeRow div.editable').on('change contentChange', 'tr div.openEndRowSettings div.openEndTypeRow div.editable', function () {
            var _row = $(this).parents('tr')[0];
            var _rowSetting = _editor.getRowSettings(_row);
            _rowSetting.OpenEndRowLabel = $(this).a4richTextEditor("getContent");
            _editor.setRowSettings(_rowSetting, _row);
        });

        // Column label update
        _self.off('change', 'tr div.gridColumnLabel').on('change', 'tr div.gridColumnLabel', function () {
            var tr = $(this).closest('tr');
            var row = _editor.getRowSettings(tr);
            _editor.fixRowLabels(row);

            var column = _editor.getColumnSettings($(this).closest('[data-column]'));
            if (column) {
                if (row.RowType == GridEditor.textRowType) {
                    var choiceIndex = $(this).closest('[data-choice-index]').attr('data-choice-index') || 0;
                    row.Labels[column.Position][choiceIndex] = $(this).a4richTextEditor("getContent");
                } else {
                    row.Labels[column.Position] = $(this).a4richTextEditor("getContent");
                }
            }
            _editor.setRowSettings(row, tr);
        });

        // Variable name update
        _self.off('change', 'tr div.gridVariable').on('change', 'tr div.gridVariable', function () {
            var _rowElement = $(this).parents('tr')[0];
            var _colElement = $(this).parents('td')[0];

            _editor.CanRenameVariables = false;
            var _row = _editor.getRowSettings(_rowElement);
            if (_row) {
                _row.VariableNames[_editor.getColumnIndex(_colElement)] = $(this).a4richTextEditor("getContent");
                _editor.setRowSettings(_row, _rowElement);
            }
        });

        // Advance settings update
        $(getQuestionEditPanel(_self)).off('change contentChange', '.v-translatableColumnSetting').on('change contentChange', '.v-translatableColumnSetting', function () {
            var settings = _editor.getColumnSettings();
            var settingValue = a4.getInputValue($(this));
            settings[0].AxisTranslatableSettings = settings[0].AxisTranslatableSettings || {};
            if (settingValue != settings[0].AxisTranslatableSettings[$(this).attr('data-name')]) {
                settings[0].AxisTranslatableSettings[$(this).attr('data-name')] = settingValue;
                _editor.setColumnSettings(settings);
            }
        });

        $(getQuestionEditPanel(_self)).off('change contentChange', '.v-columnSetting').on('change', '.v-columnSetting', function () {
            var settings = _editor.getColumnSettings();
            var settingValue = a4.getInputValue($(this));
            if (settingValue != settings[0][$(this).attr('data-name')]) {
                settings[0][$(this).attr('data-name')] = settingValue;
                _editor.setColumnSettings(settings);
            }
        });

        $(getQuestionEditPanel(_self)).off('change contentChange', '.v-columnTypeSetting').on('change', '.v-columnTypeSetting', function () {
            var settings = _editor.getColumnSettings();
            var settingValue = a4.getInputValue($(this));
            settings[0].TypeSettings = settings[0].TypeSettings || {};
            if (settingValue != settings[0].TypeSettings[$(this).attr('data-name')]) {
                settings[0].TypeSettings[$(this).attr('data-name')] = settingValue;
                _editor.setColumnSettings(settings);
            }
        });

        var getBoundingBox = function (_element) {
            var _bb = $(_element).offset();
            _bb.right = _bb.left + $(_element).outerWidth();
            _bb.bottom = _bb.top + $(_element).outerHeight();
            return _bb;
        };

        var getVerticalDragInfo = function (_position, _elements) {
            var _nearest = null;
            var _distance = Number.POSITIVE_INFINITY;

            _.each(_elements, function (_elem) {
                var _bb = getBoundingBox(_elem);

                // Compute the distance based on the center of the element. The could be negative 
                // because we want to keep track if we are above or below
                var _currentDistance = ((_bb.top + _bb.bottom) / 2.0) - _position.y;

                if (Math.abs(_currentDistance) < Math.abs(_distance)) {
                    _distance = _currentDistance;
                    _nearest = _elem;
                }
            });

            return { target: _nearest, above: _distance > 0 };
        };

        var getHorizontalDragInfo = function (_position, _elements) {
            var _nearest = null;
            var _distance = Number.POSITIVE_INFINITY;

            _.each(_elements, function (_elem) {
                var _bb = getBoundingBox(_elem);

                // Compute the distance based on the center of the element. The could be negative 
                // because we want to keep track if we are left or right
                var _currentDistance = ((_bb.left + _bb.right) / 2.0) - _position.x;

                if (Math.abs(_currentDistance) < Math.abs(_distance)) {
                    _distance = _currentDistance;
                    _nearest = _elem;
                }
            });

            return { target: _nearest, left: _distance > 0 };
        };

        ////////////////////////////////////////////////////////////////
        // Logic edition
        ////////////////////////////////////////////////////////////////
        var InitializeInlineSkipLogicEditor = function (division, logic, currentQuestionVariableOnly, hideAdvanced) {
            var gridLogicEditor = $('.v-gridLogicEditor', _self).clone(true).show();
            var logicEditor = $('.v-logicEditor', gridLogicEditor);

            if (currentQuestionVariableOnly) {
                a4logicEditor.setVariableAutoCompleteByList(logicEditor, _editor.getAllVariables());
            }

            a4logicEditor.updateVariables();
            a4logicEditor.editInlineConditionLogic(logicEditor, logic, currentQuestionVariableOnly ? null : questionnaireEditor.getInEditQuestion());
            a4logicEditor.editInlineTypeLogic(logicEditor, logic, questionnaireEditor.allowAdvanceLogic && !hideAdvanced);


            logicEditor.appendTo(division);
            return logicEditor;
        };

        var EditSkipLogic = function (logic, title, saveFn) {
            var showColumnLogicEditor = function (logic, saveFn) {
                var gridLogicEditor = $('.v-gridLogicEditor', _self).clone(true).show();
                var logicEditor = $('.v-logicEditor', gridLogicEditor);

                logicEditor.prepend('<h2>' + title + '</h2>');
                a4logicEditor.updateVariables();
                a4logicEditor.editInlineTypeLogic(logicEditor, logic, questionnaireEditor.allowAdvanceLogic);

                var content = $('<div id="EditColumnLogicContainer">').append(logicEditor.outerHTML());
                var logicResources = {
                    "Message": content
                };

                modalDialog.showConfirmDialog(logicResources,
                    function () {
                        a4logicEditor.hasError = false;

                        var oldId = (logic && logic.Id) || 0;
                        logic = a4logicEditor.getLogic(content, true);
                        logic.Id = oldId;

                        if (!a4logicEditor.hasError) {
                            saveFn(logic);
                            refreshTable();
                        }
                    }, 400, 600, null, null, {
                    initFunction: function () {
                        a4logicEditor.editInlineConditionLogic($('.v-logicEditor.updated'), logic, questionnaireEditor.getInEditQuestion())
                    }
                });
            };

            showColumnLogicEditor(logic, saveFn);
        };

        ////////////////////////////////////////////////////////////////
        // Rows operation (Edit & Delete)
        ////////////////////////////////////////////////////////////////

        // Delete row functionality
        _self.off('click', 'tr .v-linkDeleteRow')
            .on('click', 'tr .v-linkDeleteRow', function () {
                _editor.deleteRow($(this).parents('tr'));
            });

        var lastActiveRowEditionTab = 0;
        // Edit row functionality
        function CloseRowEdition(save) {
            if (save === undefined) { save = true; }

            var rowInEdition = $('tr.editing', _self);
            if (save && rowInEdition.length == 1) {
                var savedRow = _editor.getRowSettings(rowInEdition);

                if (savedRow) {
                    $('tr.rowEditor', _self).trigger('saveRequested', savedRow);
                    _editor.setRowSettings(savedRow, rowInEdition);
                }
            }

            $('tr.editing', _self).removeClass('editing');
            $('tr.rowEditor', _self).remove();
        }

        function OpenRowEditor(tr) {

            // ---------------------------------------
            // The function belows are helper to create elements in the row editor
            function createFieldLabel(text) {
                return $('<label>').addClass('fieldLabel').html(text);
            }

            function createFieldInput(type, dataName) {
                return $('<input>').addClass('v-setting fieldInput')
                    .attr({
                        'id': _.uniqueId('field'),
                        'type': type,
                        'data-name': dataName
                    });
            }

            function createSelectFieldInput(dataName, values, noSort) {
                var select = $('<select>').addClass('v-setting fieldInput').attr({ 'id': _.uniqueId('field'), 'data-name': dataName });

                var options = [];
                $.each(values, function (index, text) {
                    options.push($('<option>').attr('value', index).html(text));
                });

                if (noSort == undefined || !noSort) {
                    options.sort(function (x, y) {
                        var a = x.text().toUpperCase();
                        var b = y.text().toUpperCase();
                        return (a < b) ? -1 : (a > b) ? 1 : 0;
                    });
                }

                $.fn.append.apply(select, options);
                return select;
            }

            function createCheckBoxFieldInput(dataName) {
                return createFieldInput('checkbox', dataName);
            }

            function createTextFieldInput(dataName) {
                return createFieldInput('text', dataName);
            }

            function createDimensionSelector(dataName) {
                var container = $('<div>').addClass('fieldInput v-dimensionSelectorContainer').attr('id', _.uniqueId('field'));

                container.append($('<input>').attr('type', 'text').addClass('v-dimensionSelector v-validatePositiveInt'));
                container.append($('<select>').addClass('v-dimensionTypeSelector')
                    .append($('<option>').attr('value', 'px').text('px'), $('<option>').attr('value', '%').text('%')));
                container.append($('<input>').attr('type', 'hidden').addClass('v-setting').attr('data-name', dataName));

                return container;
            }

            function createSizeSelector() {
                var select = $('<select>').addClass('v-setting fieldInput').attr({ 'id': _.uniqueId('field'), 'data-name': 'Size' });
                select.append($('<option>').attr('value', 'Small').text(questionnaireEditor.resources.EditorField_GridOpenEnd_Size_Small));
                select.append($('<option>').attr({ 'value': 'Medium', 'selected': 'selected' }).text(questionnaireEditor.resources.EditorField_GridOpenEnd_Size_Medium));
                select.append($('<option>').attr('value', 'Large').text(questionnaireEditor.resources.EditorField_GridOpenEnd_Size_Large));
                select.append($('<option>').attr('value', 'Full').text(questionnaireEditor.resources.EditorField_GridOpenEnd_Size_Full));
                return select;
            }

            function createRichTextEditor(dataName, value) {
                var container = $('<div>').addClass('fieldInput').attr('id', _.uniqueId('field'));

                //var editable = $('<div>').addClass('editable v-qeRichEditText').attr('data-name', dataName).makeEditable({ EmptyMessage: $('.emptyEditMessage').text(), Value: value }).trigger('textChanged');
                var editable = $('<div>').html(value).addClass('editable v-qeRichEditText').attr('data-name', dataName);
                container.append(editable);

                editable.a4richTextEditor({ library: true, piping: true, inline: true });

                return container;
            }

            function createColorFieldInput(dataName) {
                return $('<div>').addClass('v-colorPicker color-container')
                    .append(createTextFieldInput(dataName).addClass('color-picker v-colorPickerValue'))
                    .append($('<div>').addClass('color-display v-colorPickerDisplay').append($('<i>')));
            }

            function appendFieldControls(div, label, input) {
                label.attr('for', input.attr('id'));
                div.append(label, input);
                return label.add(input);
            }

            // ---------------------------------------------------------------
            // Generates a tab that will contains the main row settings
            // This tabs is binded to the saveRequested to save its value into the row object, which is passed by reference
            function generateRowEditorTab(masterEditor, row) {

                // Building interface
                var rowEditorTab = $('<div>').addClass('basicRowSettings').attr('id', _.uniqueId('rowEditorTab'));

                var rowTypes = $.extend({}, questionnaireEditor.gridRowTypes);


                var parentDiv = $('<div>').addClass('gridAxisBlock').appendTo(rowEditorTab);


                var div1 = $('<div>').addClass('gridAxisEditionBlock').appendTo(parentDiv);

                appendFieldControls(div1, createFieldLabel(questionnaireEditor.resources.EditorField_RowType), createSelectFieldInput('RowType', rowTypes));
                appendFieldControls(div1, createFieldLabel(questionnaireEditor.resources.EditorField_Color), createColorFieldInput('WebColor'));
                appendFieldControls(div1, createFieldLabel(questionnaireEditor.resources.EditorField_RowDisableRandomization), createCheckBoxFieldInput('RandomizationDisabled')).addClass('titleRowSetting textRowSetting variableRowSetting');

                var div2 = $('<div>').addClass('gridAxisEditionBlock').appendTo(parentDiv);
                appendFieldControls(div2, createFieldLabel(questionnaireEditor.resources.EditorField_RowAnswerRequired), createCheckBoxFieldInput('AnswerRequired')).addClass('variableRowSetting');

                var mergeCellsCheckBox, mergeLeftLabelCheckBox, mergeRightLabelCheckBox, useChoiceTextsCheckBox;

                mergeCellsCheckBox = createCheckBoxFieldInput('MergeCells');
                appendFieldControls(div2, createFieldLabel(questionnaireEditor.resources.EditorField_MergeCells), mergeCellsCheckBox).addClass('titleRowSetting textRowSetting');
                mergeLeftLabelCheckBox = createCheckBoxFieldInput('MergeLeftLabel');
                appendFieldControls(div2, createFieldLabel(questionnaireEditor.resources.EditorField_MergeLeftLabel), mergeLeftLabelCheckBox).addClass('titleRowSetting textRowSetting');
                mergeRightLabelCheckBox = createCheckBoxFieldInput('MergeRightLabel');
                appendFieldControls(div2, createFieldLabel(questionnaireEditor.resources.EditorField_MergeRightLabel), mergeRightLabelCheckBox).addClass('titleRowSetting textRowSetting');

                if (gridCanHaveChoiceColumns()) {
                    useChoiceTextsCheckBox = createCheckBoxFieldInput('UseChoiceTexts');
                    appendFieldControls(div2, createFieldLabel(questionnaireEditor.resources.EditorField_UseChoiceTexts), useChoiceTextsCheckBox).addClass('textRowSetting');
                }

                var operationDiv = $('<div>').addClass('gridOperationBlock').appendTo(rowEditorTab);
                operationDiv.append($('<span>').addClass('fa fa-lg fa-caret-up v-linkMoveUp')
                    .attr('title', questionnaireEditor.resources.Tooltip_MoveUp),
                    $('<span>').addClass('row-delete v-linkDeleteRow')
                        .attr('title', questionnaireEditor.resources.Tooltip_DeleteRow)
                        .append($('<i>').addClass('fa fa-times')),
                    $('<span>').addClass('fa fa-lg fa-caret-down down-arrow-black v-linkMoveDown')
                        .attr('title', questionnaireEditor.resources.Tooltip_MoveDown));



                // Filling elements with data
                fillElements(rowEditorTab, row, '', '.v-setting');

                // Binding behavior
                var labelCheckBoxes = $(mergeLeftLabelCheckBox).add(mergeRightLabelCheckBox);
                mergeCellsCheckBox.on('change', function () {
                    var merge = $(this).is(':checked');
                    labelCheckBoxes.prop('disabled', !merge);
                    if (useChoiceTextsCheckBox) { useChoiceTextsCheckBox.prop('disabled', merge); }
                }).trigger('change');

                var changeSettingVisibility = function (rowType) {
                    var rowClasses = ['.variableRowSetting', '.titleRowSetting', '.textRowSetting']
                    var shownClass = '';

                    if (rowType == GridEditor.titleRowType) {
                        shownClass = rowClasses[1];
                    } else if (rowType == GridEditor.variableRowType) {
                        shownClass = rowClasses[0];
                    } else if (rowType == GridEditor.textRowType) {
                        shownClass = rowClasses[2];
                    }

                    hiddenClasses = _.filter(rowClasses, function (m) { return m != shownClass; }).join(',');

                    $(hiddenClasses, rowEditorTab).hide();
                    $(shownClass, rowEditorTab).show();
                };

                changeSettingVisibility(row.RowType || 0);
                $('.v-setting[data-name=RowType]', rowEditorTab).on("change", function () {
                    changeSettingVisibility($(this).val());
                });

                masterEditor.on('saveRequested', function (event, axis) {
                    saveElements(rowEditorTab, '.v-setting', axis);
                });

                return rowEditorTab;
            }

            // ---------------------------------------------------------------
            // Generates a tab that will skip logic
            // This tabs is binded to the saveRequested to save its value into the row object, which is passed by reference
            function generateSkipLogicTab(masterEditor, axis) {
                var hideIfEditorTab = $('<div>').addClass('skipLogicRowSettings').attr('id', _.uniqueId('rowEditorTab'));
                var logicEditor = InitializeInlineSkipLogicEditor(hideIfEditorTab, axis.HideIf);

                masterEditor.on('saveRequested', function (event, axis) {
                    var oldHideIf = axis.HideIf;
                    axis.HideIf = a4logicEditor.getLogic(logicEditor, false);

                    // Keep the id
                    if (axis.HideIf && oldHideIf && oldHideIf.Id) {
                        axis.HideIf.Id = oldHideIf.Id;
                    }
                    var tr = $("tr.editableRow[data-row=" + (axis.OriginalPosition ? axis.OriginalPosition : axis.Position) + "]", _editor.Container)
                    _editor.setRowSettings(axis, tr);
                    tr = refreshRow(tr);


                });

                return hideIfEditorTab;
            }

            function changeOpenEndSettingVisibility(oeTab, openEndType) {
                var hasOpenEnd = (openEndType != "0");
                $('.v-childOpenEndSetting', oeTab).toggle(hasOpenEnd);

                if (hasOpenEnd) {
                    tabs.tabs('option', 'disabled', [], [1]);
                }
                else {
                    if (row.RowType == GridEditor.variableRowType)
                        tabs.tabs('option', 'disabled', [2]);
                    else
                        tabs.tabs('option', 'disabled', [1, 2]);
                }
                $(".v-childOpenEndSetting.openEndTypeRow", oeTab).toggle(openEndType == "3");
            };


            // ---------------------------------------------------------------
            // Generates a tab that will contains the open end settings
            // This tabs is binded to the saveRequested to save its value into the row object, which is passed by reference
            function generateOpenEndTabs(masterEditor, row, tabs) {
                if (!row.OpenEnd) { row.OpenEnd = { EnforceOpenEnd: true, MaxLength: null }; }
                if (!row.OpenEnd.AnswerCode) { row.OpenEnd.AnswerCode = 0; }
                if (!row.OpenEnd.Lines) { row.OpenEnd.Lines = 2; }

                var oeTab = $('<div>').addClass('openEndRowSettings').attr('id', _.uniqueId('rowEditorTab'));

                // First section
                var div1 = $('<div>').addClass('gridAxisEditionBlock').appendTo(oeTab);

                if (questionnaireEditor.useS2) {
                    appendFieldControls(div1, createFieldLabel(questionnaireEditor.resources.EditorField_GridOpenEnd_Type), createCheckBoxFieldInput('Type').addClass('openEndType'));
                } else {
                    appendFieldControls(div1, createFieldLabel(questionnaireEditor.resources.EditorField_GridOpenEnd_Type), createSelectFieldInput('Type', questionnaireEditor.openEndDisplayTypes, true));
                };
                appendFieldControls(div1, createFieldLabel(questionnaireEditor.resources.EditorField_GridOpenEnd_EnforceOpenEnd), createCheckBoxFieldInput('EnforceOpenEnd')).addClass('v-childOpenEndSetting');
                appendFieldControls(div1, createFieldLabel(questionnaireEditor.resources.EditorField_GridOpenEnd_Variable), createTextFieldInput('VariableName')).addClass('v-childOpenEndSetting');
                appendFieldControls(div1, createFieldLabel(questionnaireEditor.resources.EditorField_GridOpenEnd_AnswerCode), createTextFieldInput('AnswerCode')).addClass('v-childOpenEndSetting');

                // Second section
                var div2 = $('<div>').addClass('gridAxisEditionBlock').appendTo(oeTab);
                appendFieldControls(div2, createFieldLabel(questionnaireEditor.resources.EditorField_GridOpenEnd_MinLength), createTextFieldInput('MinLength').addClass('v-validatePositiveInt')).addClass('v-childOpenEndSetting');
                appendFieldControls(div2, createFieldLabel(questionnaireEditor.resources.EditorField_GridOpenEnd_MaxLength), createTextFieldInput('MaxLength').addClass('v-validatePositiveInt')).addClass('v-childOpenEndSetting');

                if (!questionnaireEditor.useS2) {
                    appendFieldControls(div2, createFieldLabel(questionnaireEditor.resources.EditorField_GridOpenEnd_Height), createDimensionSelector('Height')).addClass('v-childOpenEndSetting');
                    appendFieldControls(div2, createFieldLabel(questionnaireEditor.resources.EditorField_GridOpenEnd_Width), createDimensionSelector('Width')).addClass('v-childOpenEndSetting');
                }
                else {
                    appendFieldControls(div2, createFieldLabel(questionnaireEditor.resources.EditorField_GridOpenEnd_Lines), createTextFieldInput('Lines').addClass('v-validatePositiveInt')).addClass('v-childOpenEndSetting');
                    appendFieldControls(div2, createFieldLabel(questionnaireEditor.resources.EditorField_GridOpenEnd_Size), createSizeSelector()).addClass('v-childOpenEndSetting');
                }

                appendFieldControls(div2, createFieldLabel(questionnaireEditor.resources.EditorField_GridOpenEnd_RowTitle), createRichTextEditor('OpenEndRowLabel', row.OpenEndRowLabel)).addClass('v-childOpenEndSetting openEndTypeRow');

                fillElements(oeTab, row.OpenEnd, '', '.v-setting');
                questionnaireEditor.updateDimensionSelectorValue(oeTab.find('.v-dimensionSelectorContainer .v-setting'));

                if (questionnaireEditor.useS2) {
                    // fillElements function can't handle the open-end convertion from 0-1 to checked correctly
                    $(".openEndType", div1).prop("checked", row.OpenEnd.Type == "1");
                }

                $('.v-setting[data-name=Type]', oeTab).on("change", function () {
                    var openEndValue = questionnaireEditor.useS2 ? ($(this).is(":checked") ? "1" : "0") : $(this).val();

                    if (openEndValue !== "0" && $('[data-name=VariableName]', oeTab).val() === "") {
                        $('[data-name=VariableName]', oeTab).val(GetQuestionName(_editor.Container).toUpperCase() + "_OE" + (row.Position + 1));
                    }
                    changeOpenEndSettingVisibility(oeTab, openEndValue);
                });

                masterEditor.on('saveRequested', function (event, axis) {
                    axis.OpenEnd = axis.OpenEnd || {};
                    saveElements(oeTab, '.v-setting', axis.OpenEnd);
                });

                return oeTab;
            }

            function generateOpenEndDisplayIfTab(masterEditor, row) {
                var oediTab = $('<div>').addClass('displayIfOpenEndSettings').attr('id', _.uniqueId('rowEditorTab'));
                var logicEditor = InitializeInlineSkipLogicEditor(oediTab, row.OpenEnd.DisplayIf, true, true);

                masterEditor.on('saveRequested', function (event, row) {
                    var oldDisplayIf = row.OpenEnd.DisplayIf;
                    row.OpenEnd.DisplayIf = a4logicEditor.getLogic(logicEditor, false);

                    // Keep the id
                    if (row.OpenEnd.DisplayIf && oldDisplayIf && oldDisplayIf.Id) {
                        row.OpenEnd.DisplayIf.Id = oldDisplayIf.Id;
                    }

                    // clear id for Display If to avoid cross referencial on copy because Ids cannot be fixed.
                    if (row.OpenEnd.DisplayIf) {
                        $.each(row.OpenEnd.DisplayIf.Conditions, function (index, element) {
                            element.VariableId = 0;
                        });
                    }

                });

                appendFieldControls(oediTab, createFieldLabel(questionnaireEditor.resources.EditorField_GridOpenEnd_DisplayIf_UseDisableInstead).addClass('fieldLabelWide'), createCheckBoxFieldInput('DisplayIfDisableInstead'));
                fillElements(oediTab, row.OpenEnd, '', '.v-setting');

                masterEditor.on('saveRequested', function (event, axis) {
                    saveElements(oediTab, '.v-setting', axis.OpenEnd);
                });

                return oediTab;
            }

            // Creates a row with only one cell.
            // It has to span over all column cells, +1 each of those: left label, right label, delete column
            CloseRowEdition();

            tr = $('tr[data-row=' + _editor.getRowIndex(tr).toString() + ']', _editor.Container);
            $(tr).addClass('editing');

            var row = _editor.getRowSettings(tr);

            var closeTabId = _.uniqueId('rowEditorTab');
            var closeButtonId = _.uniqueId('close');


            var editionRow = $('<tr>').addClass('rowEditor').attr('data-row', $(tr).attr('data-row'));
            var editionCell = $('<td>').attr('colspan', 4 + _editor.numCells()).appendTo(editionRow);


            var tabList = $('<ul>');
            var tabs = $('<div>').addClass('tabs-bottom')
                .appendTo(editionCell)
                .append(tabList);


            var rowEditorTab = generateRowEditorTab(editionRow, row);
            var openEndTab = generateOpenEndTabs(editionRow, row, tabs);
            var openEndDisplayIfTab = generateOpenEndDisplayIfTab(editionRow, row);
            var skipLogicTab = generateSkipLogicTab(editionRow, row);

            tabList.append($('<li>').append($('<a>').html(questionnaireEditor.resources.SectionTitle_Settings).attr('href', '#' + rowEditorTab.attr('id'))));
            tabs.append(rowEditorTab);

            tabList.append($('<li>').append($('<a>').html(questionnaireEditor.resources.SectionTitle_OpenEnd).attr('href', '#' + openEndTab.attr('id'))));
            tabs.append(openEndTab);

            tabList.append($('<li>').append($('<a>').html(questionnaireEditor.resources.SectionTitle_OpenEndDisplayIf).attr('href', '#' + openEndDisplayIfTab.attr('id'))));
            tabs.append(openEndDisplayIfTab);

            tabList.append($('<li>').append($('<a>').html(questionnaireEditor.resources.SectionTitle_HideIf).attr('href', '#' + skipLogicTab.attr('id'))));
            tabs.append(skipLogicTab);

            tabList.append($('<li>').attr('id', closeButtonId).addClass('closeButton').append($('<a>').html(sharedResources.Close).attr('href', '#' + closeTabId)));
            tabs.append($('<div>').attr('id', closeTabId));

            var disableTabsFromSettings = function (row) {
                if (row.RowType == GridEditor.variableRowType) {
                    openEndType = $('.v-setting[data-name=Type]', openEndTab).val();
                    if (openEndType != "0") {
                        tabs.tabs('option', 'disabled', [], [1]);
                    }
                    else {
                        tabs.tabs('option', 'disabled', [2]);
                    }
                } else {
                    if (tabs.tabs('option', 'selected') == 1) {
                        tabs.tabs('option', 'selected', 0);
                    }
                    tabs.tabs('option', 'disabled', [1, 2]);
                }
            };

            // Adding & setting up the tabs
            $(tr).after(editionRow);
            tabs.tabs({
                selected: lastActiveRowEditionTab,
                select: function (event, ui) {
                    if (ui.index !== 3) { lastActiveRowEditionTab = ui.index; }
                }
            });

            disableTabsFromSettings(row);
            changeOpenEndSettingVisibility(openEndTab, row.OpenEnd.Type || 0);

            //choose hideIf if user click on it 
            if (mouseOverHideIf)
                tabs.tabs('option', 'active', 3);

            $(".tabs-bottom, .tabs-bottom .ui-tabs-nav, .tabs-bottom .ui-tabs-nav > *", editionRow).removeClass("ui-corner-all ui-corner-top");
            $(".tabs-bottom .ui-tabs-nav > *", editionRow).removeClass("ui-corner-all ui-corner-top").addClass("ui-corner-bottom");
            $('#' + closeButtonId).css('margin-left', '40px').children('a').on("click", function () {
                CloseRowEdition();
            });

            // Hack for color pickers
            //$('.colorPickerValue', editionRow).val(_column.WebColor);
            a4.setColorPickers($('.v-colorPicker', editionRow), $(editionRow).closest('body'));

            // Binding behavior on the tabs
            $('.v-setting[data-name]', rowEditorTab).on("change", function () {
                var currentRow = _editor.getRowSettings(tr);
                var currentRowIndex = _editor.getRowIndex(tr);
                var oldRow = {};
                _.extend(oldRow, currentRow);

                editionRow.trigger('saveRequested', currentRow);

                var rowTypeChanged = oldRow.RowType != currentRow.RowType;
                if (rowTypeChanged && currentRow.RowType == GridEditor.variableRowType) {
                    if (_.size(currentRow.VariableNames) !== _.size(_editor.getColumnSettings()) ||
                        _.size(_.compact(currentRow.VariableNames)) !== _.size(_.compact(_editor.getColumnSettings()))) {
                        currentRow.VariableNames = _.map(_editor.getColumnSettings(), function (column, columnIndex) {
                            if (column) {
                                var columnCode = _editor.generateColumnCode(column, columnIndex);
                                return _editor.generateVariableName(columnCode, currentRowIndex);
                            } else {
                                return null;
                            }
                        });
                    }
                } else if (rowTypeChanged && (currentRow.RowType == GridEditor.textRowType || currentRow.RowType == GridEditor.titleRowType)) {
                    _editor.fixRowLabels(currentRow);
                }

                _editor.setRowSettings(currentRow, tr);

                disableTabsFromSettings(currentRow);

                var refresh = rowTypeChanged
                    || oldRow.MergeCells != currentRow.MergeCells
                    || oldRow.MergeLeftLabel != currentRow.MergeLeftLabel
                    || oldRow.MergeRightLabel != currentRow.MergeRightLabel
                    || oldRow.UseChoiceTexts != currentRow.UseChoiceTexts;

                if (refresh) {
                    tr = refreshRow(tr);
                }
            });

            return tabs;
        }

        _self.off('click', 'table.previewMode tr.editableRow:not(.editing)')
            .on('click', 'table.previewMode tr.editableRow:not(.editing)', function () {
                // Validates that the element is still member of the table
                var row = $(this).closest('tr.editableRow');
                if (row.parents(".gridEditor").length > 0) {
                    OpenRowEditor(row);
                }
            });
        _self.off('click', 'table.previewMode tr.editableRow.editing span.v-linkEditRowLogic')
            .on('click', 'table.previewMode tr.editableRow.editing span.v-linkEditRowLogic', function () {
                var _row = $(this).closest('tr.editableRow');
                var _editorRow = $('tr.rowEditor[data-row="' + _row.data('row') + '"]');
                if (mouseOverHideIf)
                    $('div.tabs-bottom', _editorRow).tabs('option', 'active', 3);
            })
        _self.off('click', 'table.previewMode tr.rowEditor .v-linkMoveUp')
            .on('click', 'table.previewMode tr.rowEditor .v-linkMoveUp', function () {
                var row = $(this).closest('[data-row]');
                _editor.moveRow(row, _editor.getRowPosition(row) - 1);
                refreshTable();
            });

        _self.off('click', 'table.previewMode tr.rowEditor .v-linkMoveDown')
            .on('click', 'table.previewMode tr.rowEditor .v-linkMoveDown', function () {
                var row = $(this).closest('[data-row]');
                _editor.moveRow(row, _editor.getRowPosition(row) + 1);
                refreshTable();
            });


        _self.off('mouseenter', 'tr.editableRow')
            .on('mouseenter', 'tr.editableRow', function () {
                var _row = this;
                var _deleteIcon = $(_row).find('.v-linkDeleteRow');

                if (_deleteIcon.length === 0) {
                    _deleteIcon = $('<span>').addClass('row-delete v-linkDeleteRow')
                        .attr('title', questionnaireEditor.resources.Tooltip_DeleteRow)
                        .append($('<i>').addClass('fa fa-times'));
                    $(_row).find('td.columnDeleteRow').append(_deleteIcon);
                }

                _deleteIcon.stop(true, true).fadeIn();
            });

        _self.off('mouseleave', 'tr.editableRow')
            .on('mouseleave', 'tr.editableRow', function () {
                $(this).find('.v-linkDeleteRow').stop(true, true).fadeOut(100);
            });

        _self.off('mouseenter', 'span.v-linkEditRowLogic ')
            .on('mouseenter', 'span.v-linkEditRowLogic ', function () {
                mouseOverHideIf = true;
            });
        _self.off('mouseleave', 'span.v-linkEditRowLogic ')
            .on('mouseleave', 'span.v-linkEditRowLogic', function () {
                mouseOverHideIf = false;
            });
        ////////////////////////////////////////////////////////////////
        // Choice Grid Column Type
        ////////////////////////////////////////////////////////////////
        var updateChoiceGridOptions = function (_panel, _column, value) {
            if (_column) {
                _column.ColumnType = value;
                _editor.setColumnSettings([_column]);
                refreshTable();

                // ImageSelector for ImageChoiceGrid also uses this setting
                var isRadio = value == "RadioButton";

                $(".v-minMaxContainer", _panel).toggle(!isRadio);

                if (isRadio) {
                    $(".v-minMaxContainer input", _panel).val("");
                }
            }
        };

        if (isChoiceGridType()) {
            $(getOptionPanel(_self)).off('optionsLoaded.gridEditor')
                .on('optionsLoaded.gridEditor', function () {
                    var _column = _.first(_editor.getColumnSettings());
                    updateChoiceGridOptions(getOptionPanel(this), _column, _column.ColumnType);
                });

            $('.choice-grid-answer-type input[type=radio]', _self).off('change.gridEditor').on('change.gridEditor', function () {
                var _column = _.first(_editor.getColumnSettings());
                updateChoiceGridOptions(getOptionPanel(this), _column, $(this).val());
            });
        }

        // Edit cell logic functionality
        _self.off('click', 'td.editableCell span.v-linkEditCellLogic')
            .on('click', 'td.editableCell span.v-linkEditCellLogic', function () {
                var cell = $(this).parents('[data-column]');
                var index = cell.attr("data-column");
                var indexRow = $(this).closest('tr').attr("data-row");
                var row = $("tr[data-row=" + Math.abs(indexRow) + "]");
                var rs = _editor.getRowSettings(row);

                var logic = null;
                var saveFn = null;

                if (rs) {
                    var hideIfs = rs.VariableHideIfs || [];

                    // Resize array
                    while (index > hideIfs.length) {
                        hideIfs.push(null);
                    }

                    var logic = hideIfs[index];

                    var saveFn = function (logic) {
                        hideIfs[index] = logic;
                        rs.VariableHideIfs = hideIfs;
                        _editor.setRowSettings(rs, row[0]);
                    };
                }

                EditSkipLogic(logic, questionnaireEditor.resources.CellHideIf, saveFn);
            });

        ////////////////////////////////////////////////////////////////
        // Header icons
        ////////////////////////////////////////////////////////////////

        // Delete column functionality
        _self.off('click', 'tr.v-columnOperations [data-column] .v-linkDeleteColumn',)
            .on('click', 'tr.v-columnOperations [data-column] .v-linkDeleteColumn', function () {
                var _cell = $(this).parents('[data-column]');
                if (isChoiceGridType()) {
                    var _column = _.first(_editor.getColumnSettings());
                    _column.Choices.splice(_cell.attr('data-choice-index'), 1);

                    if (isNetPromoter()) {
                        var position = 0;
                        _.each(_column.Choices, function (choice) { choice.Code = position++; });
                    }

                    _editor.setColumnSettings([_column]);
                    refreshTable();
                    refreshClientAutoCompletes();
                } else {
                    $(_cell).closest('table').find(getColumnSelector('', _cell)).remove();
                    _editor.deleteColumn(_cell);
                }
            });

        // Edit column logic functionnality
        _self.off('click', 'tr.v-columnOperations [data-column] .v-linkEditColumnLogic,span.v-linkEditColumnLogic')
            .on('click', 'tr.v-columnOperations [data-column] .v-linkEditColumnLogic,span.v-linkEditColumnLogic', function () {
                var _cell = $(this).parents('[data-column]');
                var _column = _editor.getColumnSettings(_cell);

                var _logic = null;
                var _saveFn = null;
                if (_.isString(_cell.attr('data-choice-index'))) {
                    var _index = _cell.attr('data-choice-index');
                    _logic = _column.Choices[_index].HideIf;
                    _saveFn = function (_logic) {
                        _column.Choices[_index].HideIf = _logic;
                        _editor.setColumnSettings(_column, _cell);
                    };
                }
                else {
                    _logic = _column.HideIf;
                    _saveFn = function (_logic) {
                        _column.HideIf = _logic;
                        _editor.setColumnSettings(_column, _cell);
                    };
                }

                EditSkipLogic(_logic, questionnaireEditor.resources.ColumnHideIf, _saveFn);
            });


        _self.off('click', 'tr.v-columnOperations [data-column]:not([data-choice-index]) .v-linkEditColumn')
            .on('click', 'tr.v-columnOperations [data-column]:not([data-choice-index]) .v-linkEditColumn', function () {
                var cell = $(this).parents('[data-column]');
                var basePath = $("#ApplicationBaseUrl").val();
                modalDialog.showWindow(basePath + "Survey/Questionnaire/GridColumnEditor?surveyId=" + $("#SurveyId").val() + "&columnIndex=" + cell.attr("data-column") + "&useS2=" + questionnaireEditor.useS2, 500, 820);
            });

        _self.off('click', 'tr.v-columnOperations [data-column][data-choice-index] .v-linkEditColumn',)
            .on('click', 'tr.v-columnOperations [data-column][data-choice-index] .v-linkEditColumn', function () {
                var cell = $(this).parents('[data-column]');
                var choiceIndex = cell.attr('data-choice-index');
                var basePath = $("#ApplicationBaseUrl").val();
                modalDialog.showWindow(basePath + "Survey/Questionnaire/GridColumnChoiceEditor?surveyId=" + $("#SurveyId").val() + "&choiceIndex=" + choiceIndex, 500, 820);
            });

        // Dynamic positioning of the edit & delete icons
        _self.off('mouseenter', 'tr [data-column]' + (isChoiceGridType() ? '[data-choice-index]' : ''))
            .on('mouseenter', 'tr [data-column]' + (isChoiceGridType() ? '[data-choice-index]' : ''), function () {
                var ownerTable = $(this).closest('table');
                var headerCell = $('tr.v-columnOperations ' + (isChoiceGridType() ? getColumnChoiceSelector('', this) : getColumnSelector('', this)), ownerTable);

                if (headerCell.length) {
                    var div = $(headerCell).find('div.showOnHover');
                    var _editColumn = $("span.v-linkEditColumn", headerCell);
                    var _editColumnlogic = $("span.v-linkEditColumnLogic", headerCell);
                    var position = $(headerCell).offset();

                    if (div.length === 0) {
                        div = $('<div class="showOnHover">').append('<span class="column-edit v-linkEditColumn " title="' + questionnaireEditor.resources.EditColumn + '"  ><i class="fa fa-pencil" style="font-size: 15px;"></i></span>');

                        //Remove logic edition when creating a grid
                        if (!_editor.CanRenameVariables) {
                            div.append('<span class="column-logic-edit v-linkEditColumnLogic" title="' + questionnaireEditor.resources.EditColumnLogic + '"   ><i class="fa fa-eye" style="font-size: 15px;"></i></span>');
                        }

                        div.append('<span class="column-delete v-linkDeleteColumn" title="' + questionnaireEditor.resources.DeleteColumn + '"   ><i class="fa fa-times" style="font-size: 15px;"></i></span>')
                            .css({ 'display': 'block', 'z-index': $(headerCell).findZIndex() + 1, 'text-align': 'center', 'position': 'relative', 'min-width': '70px' });
                        $(headerCell).append(div);
                    }
                    else if (div.length > 0 && _editColumn.length === 0 && _editColumnlogic.length > 0) {
                        _editColumnlogic.before('<span class="column-edit v-linkEditColumn " title="' + questionnaireEditor.resources.EditColumn + '"  ><i class="fa fa-pencil" style="font-size: 15px;"></i></span>')
                        _editColumnlogic.after('<span class="column-delete v-linkDeleteColumn" title="' + questionnaireEditor.resources.DeleteColumn + '"   ><i class="fa fa-times" style="font-size: 15px;"></i></span>')

                    }
                    var _spans = $("span:not(.showAllTime)", div);
                    _spans.stop(true, true).css('visibility', 'visible').show(); // Needed for the position call to work
                    div.position({
                        of: headerCell,
                        collision: 'none'
                    });
                    $('div.showOnHover', headerCell).css('z-index', '1');
                    _spans.stop(true, true).css({
                        'visibility': 'visible',
                        'opacity': 1,
                        'transition-delay': '0s'
                    }).fadeIn();
                }
            });

        _self.off('mouseleave', 'tr [data-column]')
            .on('mouseleave', 'tr [data-column]', function () {
                var ownerTable = $(this).closest('table');
                var headerCell = $('tr.v-columnOperations ' + (isChoiceGridType() ? getColumnChoiceSelector('', this) : getColumnSelector('', this)), ownerTable);
                $('div.showOnHover', headerCell).css('z-index', '0');
                $(headerCell).find('div.showOnHover span:not(.showAllTime)').css({
                    'visibility': 'hidden',
                    'opacity': 0,
                    'transition': 'visibility 0s linear 0.05s,opacity 0.05s linear'
                });
            });


        // TODO: Clean-up dragging code. Needs to be more generic

        /////////////////////////////////////////////////////////////////
        // Dragging rows
        /////////////////////////////////////////////////////////////////
        _self.off('mousedown', 'table.previewMode tr.editableRow .verticalGridMoveHandle',)
            .on('mousedown', 'table.previewMode tr.editableRow .verticalGridMoveHandle', function (_mouseDownArgs) {

                if (!$(_mouseDownArgs.target).is('input,textarea,select,.sliderPreview,.v-qeRichEditText.inEdit,.richEditContent') && !$(_mouseDownArgs.target).parents('.sliderPreview,.v-qeRichEditText').length) {
                    $("table.ui-grid-editor", _editor.Container).disableSelection();

                    var _row = $(this).closest('tr');

                    var _lastDragInfo = null;
                    var _cloneItem = null;
                    var _dragStartPosition = null;

                    var changeRowPosition = function (_dragInfo) {
                        if (_row !== _dragInfo.target && _row !== (_dragInfo.above ? $(_dragInfo.target).prev()[0] : $(_dragInfo.target).next()[0])) {
                            _editor.moveRow(_row, _editor.getRowPosition(_dragInfo.target) + (_dragInfo.above ? 0 : 1), true);
                            refreshTable();
                        }
                    };

                    var cleanup = function () {

                        $(document).off('.dragEvents');
                        _table.enableSelection();

                        if (_cloneItem) {
                            $(_cloneItem).remove();
                        }

                        if (_lastDragInfo) {
                            $(_lastDragInfo.target).removeClass(_lastDragInfo.above ? 'draggingAbove' : 'draggingBelow');
                            changeRowPosition(_lastDragInfo);
                        }

                    };

                    $(document).on('mousemove.dragEvents', function (_args) {

                        if (!_dragStartPosition) {
                            _dragStartPosition = { x: _args.pageX - $(_row).offset().left, y: _args.pageY - $(_row).offset().top };
                        } else {

                            if (!_cloneItem) {
                                _cloneItem = $('<table class="ui-grid-editor ui-draggable-dragging">').css({ 'position': 'absolute', 'z-index': _table.findZIndex() + 1, 'top': '0', 'left': '0', 'width': $(_row).width(), 'height': $(_row).height() });
                                _cloneItem.append($(_row).clone(false));
                                $('body').append(_cloneItem);
                            }
                            _cloneItem.css({ left: _args.pageX - _dragStartPosition.x, top: _args.pageY - _dragStartPosition.y });

                            var _dragInfo = getVerticalDragInfo({ x: _args.pageX, y: _args.pageY }, $('tr.editableRow', _table));
                            if (_dragInfo) {
                                if (_lastDragInfo && (_lastDragInfo.above != _dragInfo.above || _lastDragInfo.target != _dragInfo.target)) {
                                    $(_lastDragInfo.target).removeClass(_lastDragInfo.above ? 'draggingAbove' : 'draggingBelow');
                                    $(_dragInfo.target).addClass(_dragInfo.above ? 'draggingAbove' : 'draggingBelow');
                                }
                                _lastDragInfo = _dragInfo;
                            }
                        }

                    });

                    $(document).on('mouseup.dragEvents', function () {
                        cleanup();
                    });
                }

            });

        /////////////////////////////////////////////////////////////////
        // Dragging columns
        /////////////////////////////////////////////////////////////////
        _self.off('mousedown', 'tr.v-columnOperations [data-column]:not([data-choice-index])')
            .on('mousedown', 'tr.v-columnOperations [data-column]:not([data-choice-index])', function (_mouseDownArgs) {
                if (!$(_mouseDownArgs.target).is('input,textarea,select,.sliderPreview,.v-qeRichEditText.inEdit,.richEditContent') && !$(_mouseDownArgs.target).parents('.sliderPreview,.v-qeRichEditText').length) {
                    $("table.ui-grid-editor", _editor.Container).disableSelection();

                    var _columnHeader = this;
                    var _columnIndex = $(this).attr('data-column');

                    var _lastDragInfo = null;
                    var _cloneItem = null;
                    var _dragStartPosition = { x: _mouseDownArgs.pageX - $(_columnHeader).offset().left, y: _mouseDownArgs.pageY - $(_columnHeader).offset().top };

                    var changeColumnPosition = function (_dragInfo) {
                        if (_columnIndex != $(_dragInfo.target).attr('data-column') && _columnIndex != $(_dragInfo.left ? $(_dragInfo.target).prev()[0] : $(_dragInfo.left).next()[0]).attr('data-column')) {
                            _editor.moveColumn(_columnHeader, _editor.getColumnPosition(_dragInfo.target) + (_dragInfo.left ? 0 : 1));
                            refreshTable();
                        }
                    };

                    var cleanup = function () {

                        $(document).off('.dragEvents');
                        _table.enableSelection();

                        if (_cloneItem) {
                            $(_cloneItem).remove();
                        }

                        if (_lastDragInfo) {
                            $('.draggingLeft', _table).removeClass('draggingLeft');
                            $('.draggingRight', _table).removeClass('draggingRight');
                            changeColumnPosition(_lastDragInfo);
                        }

                    };

                    $(document).on('mousemove.dragEvents', function (_args) {

                        if (_cloneItem || (Math.pow(_args.pageX - _mouseDownArgs.pageX, 2) + Math.pow(_args.pageY - _mouseDownArgs.pageY, 2)) > 50) {
                            if (!_cloneItem) {
                                _cloneItem = $('<table class="ui-grid-editor ui-draggable-dragging">').css({ 'position': 'absolute', 'z-index': _table.findZIndex() + 1, 'top': '0', 'left': '0', 'width': $(_columnHeader).width() });
                                // Add colgroup
                                _cloneItem.append(_table.find(getColumnSelector('colgroup', _columnHeader)).clone(false));

                                _cloneItem.append(_.map(_table.children('tbody').children('tr'), function (_row) {
                                    var _newRow = $('<tr>');
                                    _newRow.append($(_row).find(getColumnSelector('', _columnHeader)).clone(false));
                                    return _newRow[0];
                                }));
                                _cloneItem.find('.sprite-editor-icons').remove();
                                $('body').append(_cloneItem);
                            }
                            _cloneItem.css({ left: _args.pageX - _dragStartPosition.x, top: _args.pageY - _dragStartPosition.y });

                            var _dragInfo = getHorizontalDragInfo({ x: _args.pageX, y: _args.pageY }, $('tr:first [data-column]:not([data-choice-index])', _table));
                            if (_dragInfo) {
                                if (_lastDragInfo && (_lastDragInfo.left != _dragInfo.left || _lastDragInfo.target != _dragInfo.target)) {
                                    var _lastClass = _lastDragInfo.left ? 'draggingLeft' : 'draggingRight';
                                    $('.' + _lastClass, _table).removeClass(_lastClass);
                                }
                                $('tr', _table).each(function () {
                                    if (_dragInfo.left) {
                                        $(getColumnSelector('', _dragInfo.target), this).first().addClass('draggingLeft');
                                    } else {
                                        $(getColumnSelector('', _dragInfo.target), this).last().addClass('draggingRight');
                                    }
                                });
                                _lastDragInfo = _dragInfo;
                            }
                        }

                    });

                    $(document).on('mouseup.dragEvents', function () {
                        cleanup();
                    });
                }

            });


        /////////////////////////////////////////////////////////////////
        // Dragging choice column columns
        /////////////////////////////////////////////////////////////////
        _self.off('mousedown', 'tr.v-columnOperations [data-column][data-choice-index]')
            .on('mousedown', 'tr.v-columnOperations [data-column][data-choice-index]', function (_mouseDownArgs) {
                if (!$(_mouseDownArgs.target).is('input,textarea,select,.sliderPreview,.v-qeRichEditText.inEdit,.richEditContent') && !$(_mouseDownArgs.target).parents('.sliderPreview,.v-qeRichEditText').length) {
                    $("table.ui-grid-editor", _editor.Container).disableSelection();

                    var _columnChoice = this;
                    var _columnIndex = $(this).attr('data-column');
                    var _choiceIndex = $(this).attr('data-choice-index');

                    var _lastDragInfo = null;
                    var _cloneItem = null;
                    var _dragStartPosition = { x: _mouseDownArgs.pageX - $(_columnChoice).offset().left, y: _mouseDownArgs.pageY - $(_columnChoice).offset().top };

                    var changeColumnPosition = function (_dragInfo) {
                        if (_choiceIndex != $(_dragInfo.target).attr('data-choice-index') && _choiceIndex != $(_dragInfo.left ? $(_dragInfo.target).prev()[0] : $(_dragInfo.left).next()[0]).attr('data-choice-index')) {
                            var _newIndex = new Number($(_dragInfo.target).attr('data-choice-index'), 10) + (_dragInfo.left ? 0 : 1);
                            var displacement = _newIndex - _choiceIndex;
                            if (displacement > 0) {
                                displacement--;
                            }
                            _editor.moveChoice(_editor.getColumnIndex(_columnChoice), _choiceIndex, displacement);
                            refreshTable();
                        }
                    };

                    var cleanup = function () {

                        $(document).off('.dragEvents');

                        if (_table === undefined)
                            _table = _self.find('table.ui-grid-editor')

                        _table.enableSelection();

                        if (_cloneItem) {
                            $(_cloneItem).remove();
                        }

                        if (_lastDragInfo) {
                            $('.draggingLeft', _table).removeClass('draggingLeft');
                            $('.draggingRight', _table).removeClass('draggingRight');
                            changeColumnPosition(_lastDragInfo);
                        }

                    };

                    $(document).on('mousemove.dragEvents', function (_args) {

                        if (_cloneItem || (Math.pow(_args.pageX - _mouseDownArgs.pageX, 2) + Math.pow(_args.pageY - _mouseDownArgs.pageY, 2)) > 50) {

                            if (!_cloneItem) {
                                _cloneItem = $('<table class="ui-grid-editor ui-draggable-dragging">').css({ 'position': 'absolute', 'z-index': _table.findZIndex() + 1, 'top': '0', 'left': '0', 'width': $(_columnChoice).width() });

                                _cloneItem.append(_.map(_table.children('tbody').children('tr'), function (_row) {
                                    var _newRow = $('<tr>');
                                    _newRow.append($(_row).find(getColumnChoiceSelector('', _columnChoice)).clone(false));
                                    return _newRow[0];
                                }));
                                _cloneItem.find('.sprite-editor-icons').remove();
                                $('body').append(_cloneItem);
                            }
                            _cloneItem.css({ left: _args.pageX - _dragStartPosition.x, top: _args.pageY - _dragStartPosition.y });

                            var _dragInfo = getHorizontalDragInfo({ x: _args.pageX, y: _args.pageY }, $('tr:first [data-column=' + _columnIndex + '][data-choice-index]', _table));
                            if (_dragInfo) {
                                if (_lastDragInfo && (_lastDragInfo.left != _dragInfo.left || _lastDragInfo.target != _dragInfo.target)) {
                                    var _lastClass = _lastDragInfo.left ? 'draggingLeft' : 'draggingRight';
                                    $('.' + _lastClass, _table).removeClass(_lastClass);
                                }
                                $('tr', _table).each(function () {
                                    if (_dragInfo.left) {
                                        $(getColumnChoiceSelector('', _dragInfo.target), this).first().addClass('draggingLeft');
                                    } else {
                                        $(getColumnChoiceSelector('', _dragInfo.target), this).last().addClass('draggingRight');
                                    }
                                });
                                _lastDragInfo = _dragInfo;
                            }
                        }
                    });

                    $(document).on('mouseup.dragEvents', function () {
                        cleanup();
                    });
                }

            });

        if (isChoiceGridType()) {
            var data = _editor.getColumnSettings()[0].Choices;
            _self.find('.v-selectChoiceGridChoice').each(function () {
                $(this).a4autocomplete({
                    data: data,
                    multipleSelection: false,
                    valueProperty: "Code",
                    labelProperty: "Text"
                });
            });
        }

        if ($gridEditorOptions === "closeEdition") {
            CloseRowEdition();
        }
        else if ($gridEditorOptions === "refreshTable") {
            refreshTable();
        }
        else if ($gridEditorOptions === "validate") {
            CloseRowEdition();


            // Validate logic objects
            var valid = true;

            var rowSettings = _editor.getRowSettings();
            var columnSettings = _editor.getColumnSettings();
            var allVariables = _editor.getAllVariables();
            var variableNames = [];

            function ValidateVariableNames(names) {
                function isValid(name) {
                    name = (name || '').toUpperCase();
                    if (name.length === 0) {
                        a4.showErrorMessage(questionnaireEditor.resources.EmptyVariableNames);
                        return false;
                    } else if (_.include(variableNames, name)) {
                        a4.showErrorMessage(questionnaireEditor.resources.DuplicateVariableNames.replace("{0}", name));
                        return false;
                    } else if (!questionnaireEditor.validName(name)) {
                        a4.showErrorMessage(questionnaireEditor.resources.ErrorVariableInvalidSyntax.replace("{0}", name));
                        return false;
                    } else {
                        variableNames.push(name);
                        return true;
                    }
                }

                if (_.isString(names)) {
                    return isValid(names);
                } else {
                    return _.all(names, isValid);
                }

            }

            _.each(rowSettings, function (row, index) {
                if (row) {
                    var tr = $("[data-row=" + index + "]", _editor.Container);
                    if (valid && row.OpenEnd && row.OpenEnd.Type != "0") {
                        if (!ValidateVariableNames(row.OpenEnd.VariableName)) {
                            valid = false;
                            renderPreviewModeTable();
                            $(OpenRowEditor(tr)).scrollIntoView();
                            $('.tabs-bottom').tabs('option', 'active', 1);
                            $('.openEndRowSettings [data-name=VariableName]').focus();
                        }
                    }

                    if (valid && row.RowType == GridEditor.variableRowType && row.VariableNames) {
                        var rowNames = [];
                        _.each(columnSettings, function (value, index) {
                            if (value) {
                                rowNames.push(row.VariableNames[index]);
                            }
                        });

                        if (!ValidateVariableNames(rowNames)) {
                            valid = false;
                            renderVariableModeTable();
                        }
                    }

                    if (valid && row.OpenEnd) {
                        var min = parseInt(row.OpenEnd.MinLength);
                        var max = parseInt(row.OpenEnd.MaxLength);
                        if (!isNaN(min) && !isNaN(max) && min > max) {
                            valid = false;
                            a4.showErrorMessage(questionnaireEditor.resources.MaxValueLowerThanMinValue);
                            renderPreviewModeTable();
                            $(OpenRowEditor(tr)).scrollIntoView();
                            $('.tabs-bottom').tabs('option', 'active', 1);
                        }
                    }

                    if (valid && row.HideIf) {
                        if (!a4logicEditor.validateLogic(row.HideIf, questionnaireEditor.resources.SectionTitle_HideIf)) {
                            valid = false;
                            renderPreviewModeTable();
                            $(OpenRowEditor(tr)).scrollIntoView();
                            $('.tabs-bottom').tabs('option', 'active', 2);
                        }
                    }

                    if (valid && row.OpenEnd && row.OpenEnd.DisplayIf) {
                        if (!a4logicEditor.validateClientSideLogic(row.OpenEnd.DisplayIf, allVariables, questionnaireEditor.resources.SectionTitle_OpenEndDisplayIf)) {
                            valid = false;
                            renderPreviewModeTable();
                            $(OpenRowEditor(tr)).scrollIntoView();
                            $('.tabs-bottom').tabs('option', 'active', 2);
                        }
                    }
                }
            });

            returnValue = returnValue && valid;
        } else {
            /////////////////////////////////////////////////////////////////
            // Initializing the table
            /////////////////////////////////////////////////////////////////
            if ($('[data-name=columns]', _editor.Container).val() === '' && $('[data-name=rows]', _editor.Container).val() === '') {
                _editor.CanRenameVariables = true;
                $(GetQuestionNameControl(_editor.Container)).change(function () {
                    if (_editor.resetVariableNames() && _table.hasClass('variableMode')) {
                        refreshTable();
                    }
                });

                // New Question
                _editor.createEmptyGrid(getGridType());
            }

            $('.choice-grid-answer-type input[type=radio]', _self).each(function () {
                var _id = _.uniqueId('choiceGridType');
                $(this).attr('id', _id);
                $(this).next('label').attr('for', _id);
            });

            var _choiceType = _.first(_editor.getColumnSettings()) && _.first(_editor.getColumnSettings()).ColumnType;
            $('.choice-grid-answer-type input[value=' + _choiceType + ']', _self).prop('checked', true);

            var _table = _self.find('table.ui-grid-editor');
            CloseRowEdition(false); // Making sure there is not edition (happens when the user cancel edition)
            refreshTable();
        }
    });

    return returnValue;
};